package ru.yandex.qe.mail.meetings.synchronizer.dao;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.inject.Named;
import javax.sql.DataSource;
import javax.transaction.Transactional;

import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.stereotype.Repository;

import ru.yandex.qe.mail.meetings.synchronizer.dto.IdWithType;
import ru.yandex.qe.mail.meetings.synchronizer.dto.SourceType;
import ru.yandex.qe.mail.meetings.synchronizer.dto.SyncEvent;

@Repository
public class SyncDao {
    private static final Logger LOG = LoggerFactory.getLogger(SyncDao.class);

    private NamedParameterJdbcTemplate jdbcTemplate;

    @Autowired
    public SyncDao(@Named("psqlDataSource") DataSource psqlDataSource) {
        this.jdbcTemplate = new NamedParameterJdbcTemplate(psqlDataSource);
    }

    @Transactional
    public List<SyncEvent> getAllEvents() {
        return jdbcTemplate.query(
                " SELECT event_id, owner FROM synchronized_events",
                (rs, rowNum) -> new SyncEvent(rs.getString("owner"), rs.getInt("event_id"))
        );
    }

    @Transactional
    public Set<IdWithType> getAssignedIds(int eventId) {
        var params = new MapSqlParameterSource("event_id", eventId);
        var sql = "SELECT entity_id, type FROM synchronization_relationships where event_id = :event_id";
        return new HashSet<>(
                jdbcTemplate.query(
                        sql,
                        params,
                        (rs, rowNum) -> new IdWithType(rs.getString("entity_id"), SourceType.valueOf(rs.getString("type")))
                )
        );
    }

    @Transactional
    public void assign(SyncEvent syncEvent, Set<IdWithType> ids) {
        LOG.info("assign event with id {} and owner {} to [{}]", syncEvent.id(), syncEvent.owner(), ids);
        jdbcTemplate.update("INSERT INTO synchronized_events (event_id, owner) VALUES (:event_id, :owner)",
                Map.of(
                        "event_id", syncEvent.id(),
                        "owner", syncEvent.owner()
                )
        );

        var createRelationshipsSql = "INSERT INTO synchronization_relationships (event_id, entity_id, type) VALUES (:event_id, :entity_id, :type::source_type)";

        var params = ids
                .stream()
                .map(id ->
                        new MapSqlParameterSource("event_id", syncEvent.id())
                                .addValue("entity_id", id.id)
                                .addValue("type", "" + id.type)
                ).toArray(MapSqlParameterSource[]::new);

        jdbcTemplate.batchUpdate(createRelationshipsSql, params);
    }

    @Transactional
    public int unassign(int eventId) {
        LOG.info("unassign event with id {}", eventId);
        return jdbcTemplate.update(
                "DELETE FROM synchronized_events WHERE event_id = :event_id",
                Map.of("event_id", eventId)
        );
    }

    @Transactional
    public int unassign(int eventId, IdWithType idWithType) {
        LOG.info("unassign {}-{} from event with id {}", idWithType.id, idWithType.type, eventId);
        return jdbcTemplate.update(
                "DELETE FROM synchronization_relationships WHERE event_id=:event_id and entity_id=:entity_id and type=:type::source_type",
                Map.of(
                        "event_id", eventId,
                        "entity_id", idWithType.id,
                        "type", "" + idWithType.type
                )
        );
    }

    /**
     * Возвращает отображение event_id -> entity_wit_type для всех event_id, у которых указанный пользователь является овнером
     * @param login - логин
     */
    @Transactional
    public Map<SyncEvent, List<IdWithType>> getAllByOwner(String login) {
        return jdbcTemplate.query(
                "select * from (select * from synchronized_events where owner=:login) as a LEFT JOIN synchronization_relationships ON a.event_id=synchronization_relationships.event_id",
                new MapSqlParameterSource("login", login),
                (rs, rowNum) -> Pair.of(
                        new SyncEvent(rs.getString("owner"), rs.getInt("event_id")),
                        rs.getString("entity_id") != null ? new IdWithType(rs.getString("entity_id"), SourceType.valueOf(rs.getString("type"))) : IdWithType.NONE)
        ).stream()
                .collect(Collectors.toMap(Pair::getKey, p -> List.of(p.getValue()), (l1, l2) -> {
                    var result = new ArrayList<IdWithType>(l1.size() + l2.size());
                    result.addAll(l1);
                    result.addAll(l2);
                    return result;
                }));
    }

    @Transactional
    public Map<SyncEvent, List<IdWithType>> getAllEventsWithLinkedGroups() {
        return jdbcTemplate.query(
                "SELECT synchronized_events.event_id, synchronized_events.owner, synchronization_relationships.entity_id, synchronization_relationships.type from synchronized_events " +
                        "LEFT JOIN synchronization_relationships " +
                        "ON synchronized_events.event_id = synchronization_relationships.event_id ",
                Collections.emptyMap(),
                (rs, rowNum) -> Pair.of(
                        new SyncEvent(rs.getString("owner"), rs.getInt("event_id")),
                        rs.getString("entity_id") != null ? new IdWithType(rs.getString("entity_id"), SourceType.valueOf(rs.getString("type"))) : IdWithType.NONE)
        ).stream()
                .collect(Collectors.toMap(Pair::getKey, p -> List.of(p.getValue()), (l1, l2) -> {
                    var result = new ArrayList<IdWithType>(l1.size() + l2.size());
                    result.addAll(l1);
                    result.addAll(l2);
                    return result;
                }));
    }
}
