package ru.yandex.qe.mail.meetings.synchronizer.impl;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.qe.mail.meetings.services.staff.dto.Person;
import ru.yandex.qe.mail.meetings.synchronizer.MultipleChoiceException;

public abstract class UpdatableCache<T extends Comparable<T>> {
    private static final Logger LOG = LoggerFactory.getLogger(UpdatableCache.class);
    private static final long MIN_UPDATE_DELAY = 10000L;

    private volatile CacheState state;

    public abstract List<Person> membersById(int serviceId);

    protected abstract String name(T cachedObject, String lang);
    protected abstract Integer id(T cachedObject);
    protected abstract List<T> load();

    protected synchronized void update() {
        if (shouldSkipUpdate() || Boolean.getBoolean("lazy.cache.initialization")) {
            return;
        }
        LOG.info("updating {}", this.getClass().getSimpleName());
        var instances = load();
        updateCacheState(new CacheState(instances));
        LOG.info("updated");
    }

    public Optional<T> byName(@Nonnull String name, @Nonnull String lang) throws MultipleChoiceException {
        updateIfNeeded();

        final List<T> candidates;

        if ("ru".equals(lang)) {
            candidates = state.ruMapping.get(name.toLowerCase());
        } else {
            candidates = state.all.stream().filter(s -> name.equalsIgnoreCase(name(s, lang))).collect(Collectors.toList());
        }

        return extractSingle(candidates);
    }

    public Optional<T> byId(int objId) {
        updateIfNeeded();
        return Optional.ofNullable(state.idMapping.get(objId));
    }


    private <R> Optional<R> extractSingle(List<R> list) throws MultipleChoiceException {
        if (list == null || list.size() == 0) {
            return Optional.empty();
        }
        if (list.size() == 1) {
            return Optional.of(list.get(0));
        }
        throw new MultipleChoiceException();
    }

    private synchronized void updateIfNeeded() {
        if (state == null) {
            LOG.warn("null state, should update");
            update();
        }
    }

    protected boolean shouldSkipUpdate() {
        if (state != null && (System.currentTimeMillis() - state.creationTs) < MIN_UPDATE_DELAY) {
            LOG.info("skipping {} list update (rate limit)", this.getClass().getSimpleName());
            return true;
        }
        return false;
    }

    protected void updateCacheState(CacheState cs) {
        this.state = cs;
    }

    protected final class CacheState {
        private final Map<String, List<T>> ruMapping;
        private final Map<Integer, T> idMapping;
        private final List<T> all;

        private final long creationTs = System.currentTimeMillis();

        CacheState(@Nonnull List<T> all) {
            this.all = all.stream().sorted().distinct().collect(Collectors.toList());
            this.ruMapping = this.all.stream().collect(Collectors.toMap(
                    s -> name(s, "ru").toLowerCase(),
                    Collections::singletonList,
                    (a, b) -> Stream.concat(a.stream(), b.stream()).collect(Collectors.toList()))
            );
            this.idMapping = this.all.stream().collect(Collectors.toMap(UpdatableCache.this::id, s->s, (a, b) -> a));
        }
    }
}
