package ru.yandex.qe.mail.meetings.utils;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.qe.mail.meetings.services.calendar.CalendarWeb;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Event;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Response;
import ru.yandex.qe.mail.meetings.services.calendar.dto.faults.CalendarException;

/**
 * @author Sergey Galyamichev
 */
@Component
public class CalendarHelper {
    private static final Logger LOG = LoggerFactory.getLogger(CalendarHelper.class);

    private final CalendarWeb calendarWeb;

    public CalendarHelper(CalendarWeb calendarWeb) {
        this.calendarWeb = calendarWeb;
    }

    /**
     * We need base event to store DB - occurrence may expire
     *
     * @param eventId - occurrence ID
     * @return event or base event for series
     */
    public Event getBaseEvent(Integer eventId) {
        Event occurrence = calendarWeb.getEvent(eventId);
        return getBaseEvent(occurrence);
    }

    /**
     * We need base event to store DB - occurrence may expire
     *
     * @param event - occurrence
     * @return event or base event for series
     */
    public Event getBaseEvent(Event event) {
        if (event.getIsRecurrence()) {
            event = calendarWeb.getEvent(event.getEventId(), null, true);
        }
        return event;
    }

    /**
     * Tries to find clean series instance to update
     *
     * @param eventId occurrence or series id
     * @return event series instance without overriding event
     */
    public Event getSeriesInstance(int eventId) {
        Event event = calendarWeb.getEvent(eventId);
        boolean isOccurrence = event.getIsRecurrence();
        if (isOccurrence) {
            event = tryToFindMainEvent(calendarWeb.getEvent(event.getEventId(), null, true));
            if (event == null) {
                throw new CalendarException("Unable to find main event", "depth limit exceeded");
            }
        }
        return event;
    }

    private Event tryToFindMainEvent(Event event) {
        List<Event> occurrences = new ArrayList<>();
        String instanceStartTs = event.getInstanceStartTs();
        Event candidate;
        try {
            do {
                candidate = calendarWeb.getEvent(event.getEventId(), instanceStartTs, false);
                occurrences.add(candidate);
                if (candidate.getRepetition() == null) {
                    break;
                }
                instanceStartTs = RepetitionUtils.next(instanceStartTs, event.getRepetition());
            } while (candidate.getIsRecurrence() && occurrences.size() < 100);
        } catch (Exception e) {
            throw new CalendarException("Unable to find main event", e.getMessage(), e);
        }
        return candidate;
    }

    /**
     * Wrap call of calendar service and throws <code>CalendarException<code/> if something goes wrong
     *
     * @param remote - method calendar
     */
    public void doCall(Supplier<Response> remote) {
        Response response = remote.get();
        if (!response.isOk()) {
            throw new CalendarException(response.getStatus(), response.toString());
        }
    }

    public static boolean isBigEvent(Event event) {
        int attendees = event.getAttendees().size();
        if (attendees > 10) {
            LOG.info("Big event skipped {}", attendees);
            return true;
        }
        return false;
    }

    public static boolean isVeryBigEvent(Event event) {
        int attendees = event.getAttendees().size();
        if (attendees > 200) {
            LOG.info("Very big event skipped {}", attendees);
            return true;
        }
        return false;
    }
}
