package ru.yandex.qe.mail.meetings.utils;

import java.util.Calendar;
import java.util.Date;
import java.util.TimeZone;

public class DateRange {
    private final Date from;
    private final Date to;

    private DateRange(Date from, Date to) {
        this.from = from;
        this.to = to;
    }

    public Date getFrom() {
        return from;
    }

    public Date getTo() {
        return to;
    }

    public boolean contains(Date date) {
        return from.getTime() <= date.getTime() && date.getTime() < to.getTime();
    }

    public long duration() {
        return to.getTime() - from.getTime();
    }

    public static DateRange range(Date from, Date to) {
        if (to.getTime() < from.getTime()) {
            throw new IllegalArgumentException("'to' shall be greater or equal to 'from'");
        }
        return new DateRange(from, to);
    }

    public static void toStartOfADay(Calendar c) {
        c.set(Calendar.MILLISECOND, 0);
        c.set(Calendar.SECOND, 0);
        c.set(Calendar.MINUTE, 0);
        c.set(Calendar.HOUR_OF_DAY, 0);
    }

    public static DateRange nextBusinessWeek(Date date) {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        int today = c.get(Calendar.DAY_OF_WEEK);
        int shift = today == Calendar.SATURDAY || today == Calendar.SUNDAY ? 7 : 14;
        c.add(Calendar.DAY_OF_MONTH, shift);
        toStartOfADay(c);
        return lastBusinessWeek(c.getTime());
    }

    public static DateRange lastBusinessWeek(Date date) {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        int today = c.get(Calendar.DAY_OF_WEEK);
        today += today > Calendar.FRIDAY ? 0 : 7;
        c.add(Calendar.DAY_OF_MONTH, Calendar.FRIDAY - today);
        Date to = c.getTime();
        c.add(Calendar.DAY_OF_MONTH, Calendar.MONDAY - Calendar.FRIDAY);
        Date from = c.getTime();
        return range(from, to);
    }

    public static Date fromUTCtoMSK(Date utcDate) {
        return convert(utcDate, TimeZone.getTimeZone("UTC"), TimeZone.getTimeZone("Europe/Moscow"));
    }

    public static Date toMSK(Date date) {
        return convert(date, TimeZone.getDefault(), TimeZone.getTimeZone("Europe/Moscow"));
    }

    public static Date fromUTC(Date date) {
        return convert(date, TimeZone.getTimeZone("UTC"), TimeZone.getDefault());
    }

    public static Date convert(Date utcDate, TimeZone from, TimeZone to) {
        long timeDifference = to.getRawOffset() - from.getRawOffset() + to.getDSTSavings() - from.getDSTSavings();
        Calendar calendar = Calendar.getInstance(from);
        calendar.setTime(utcDate);
        calendar.add(Calendar.MILLISECOND, (int) timeDifference);
        return calendar.getTime();
    }

    public static Date tomorrow(Date date) {
        return addInterval(date, Calendar.DAY_OF_MONTH, 1);
    }

    public static Date addInterval(Date eventStart, int timeUnit, int quantity) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(eventStart);
        calendar.add(timeUnit, quantity);
        return calendar.getTime();
    }

    public static Date today() {
        Calendar c = Calendar.getInstance();
        toStartOfADay(c);
        return c.getTime();
    }
}
