package ru.yandex.qe.mail.meetings.utils;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.springframework.stereotype.Component;

import ru.yandex.qe.mail.meetings.api.resource.dto.ActionType;
import ru.yandex.qe.mail.meetings.api.resource.dto.CalendarAction;
import ru.yandex.qe.mail.meetings.cron.AbstractMessageBuilder;
import ru.yandex.qe.mail.meetings.services.calendar.CalendarWeb;

/**
 * @author Sergey Galyamichev
 */
@Component
public class HtmlBuilder {
    private final CalendarWeb calendarWeb;

    public HtmlBuilder(CalendarWeb calendarWeb) {
        this.calendarWeb = calendarWeb;
    }


    public static String toPretty(List<CalendarAction> actions) {
        if (actions.isEmpty()) {
            return "<div>Заявок нет</div>";
        }
        return "<table>" + actions.stream()
                .map(HtmlBuilder::toPretty)
                .collect(Collectors.joining()) + "</table>";
    }

    private static String toPretty(CalendarAction calendarAction) {
        return "<tr>" +
                "<td>" + calendarLink(calendarAction.getEventId()) + "</td>" +
                "<td>" + staffLink(AbstractMessageBuilder.getLogin(calendarAction.getEmail())) + "</td>" +
                "<td>" + calendarAction.getStart()+"</td>" +
                "</tr>";
    }

    private static String calendarLink(int eventId) {
        return "<a href=\"https://calendar.yandex-team.ru/event/" + eventId +"\">" + eventId + "</a>";
    }

    private static String staffLink(String login) {
        return "<a href=\"https://staff.yandex-team.ru/" + login +"\">" + login + "</a>";
    }

    public String buildResponse(List<CalendarAction> actions) {
        Map<ActionType, List<CalendarAction>> map = actions.stream()
                .collect(Collectors.groupingBy(CalendarAction::getType));
        StringBuilder result = new StringBuilder();
        for (Map.Entry<ActionType, List<CalendarAction>> entry : map.entrySet()) {
            result.append(buildSection(getCaption(entry.getKey()), entry.getValue()));
        }
        return result.toString();
    }

    private static String getCaption(ActionType key) {
        switch (key) {
            case REMOVE_EVENT:
                return "Вы отменили следующие встречи:";
            case DECLINE_EVENT:
                return "Вы отказались от следующих встреч:";
            default:
                return "Вы изменили следующие встречи:";
        }
    }

    public String buildSection(String caption, List<CalendarAction> actions) {
        String body = actions.stream()
                .map(a -> actionRow(calendarWeb, a))
                .collect(Collectors.joining());
        return "<h3>" + caption + "</h3>" +
                "<table>" +
                    header() +
                    body +
                "</table>" +
                "<h3>Спасибо, что сделали календарь актуальным<h4>" +
                "<h4>Обратную связь можно написать на <a href=\"mailto:meetings-report@yandex-team.ru\">meetings-report@</a></h4>";
    }

    private static String header() {
        return "<tr>" +
                "<th>Id</th>" +
                "<th>Название</th>" +
                "<th>Дата начала</th>" +
                "</tr>";
    }


    private static String actionRow(CalendarWeb calendarWeb, CalendarAction calendarAction) {
        return "<tr>" +
                "<td>" + calendarLink(calendarAction.getEventId()) + "</td>" +
                "<td>" + getName(calendarWeb, calendarAction) + "</td>" +
                "<td>" + calendarAction.getStart()+"</td>" +
                "</tr>";
    }

    private static String getName(CalendarWeb calendarWeb, CalendarAction calendarAction) {
        try {
            return calendarWeb.getEvent(calendarAction.getEventId()).getName();
        } catch (Exception e) {
            return "Unknown";
        }
    }
}
