package ru.yandex.qe.mail.meetings.utils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import org.springframework.stereotype.Component;

import ru.yandex.qe.mail.meetings.analisys.EventGapAnalyzer;
import ru.yandex.qe.mail.meetings.cron.actions.Contexts;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Event;
import ru.yandex.qe.mail.meetings.services.calendar.dto.EventDate;
import ru.yandex.qe.mail.meetings.services.calendar.dto.EventUser;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Resource;
import ru.yandex.qe.mail.meetings.services.gaps.GapApi;
import ru.yandex.qe.mail.meetings.services.gaps.dto.Gap;
import ru.yandex.qe.mail.meetings.services.gaps.dto.Gaps;
import ru.yandex.qe.mail.meetings.services.staff.StaffClient;
import ru.yandex.qe.mail.meetings.services.staff.dto.Person;

/**
 * @author Sergey Galyamichev
 */
@Component
public class OfficeSplitter {
    public static final int ABSENT = -2;
    public static final int IN_TRIP = -1;

    private static final Set<String> TRUE_ABSENCE = Set.of("absence", "vacation", "paid_day_off", "illness", "maternity");
    private static final Set<String> TRIP = Set.of("trip", "conference_trip");
    private final StaffClient staffClient;
    private final GapApi gapApi;

    public OfficeSplitter(StaffClient staffClient, GapApi gapApi) {
        this.staffClient = staffClient;
        this.gapApi = gapApi;
    }

    public Map<Integer, List<Person>> split(Event event, Person requester) {
        List<String> users = new ArrayList<>(getLogins(event.getAttendees()));
        users.add(event.getOrganizer() != null ? event.getOrganizer().getLogin() : requester.getLogin());
        return split(event, users);
    }

    private List<String> getLogins(List<EventUser> users){
        return users.stream()
                .map(EventUser::getLogin)
                .filter(Objects::nonNull) // visitor doesn't have login
                .collect(Collectors.toList());
    }

    public Map<Integer, List<Person>> split(EventDate event, List<String> logins) {
        Map<String,List<Gap>> gaps = getGaps(event, logins);
        return logins.stream()
                .map(staffClient::getByLogin)
                .collect(Collectors.groupingBy(person -> getOfficeId(gaps, person, event)));
    }

    private Map<String,List<Gap>> getGaps(EventDate event, List<String> logins) {
        Map<String, List<Gap>> result = new HashMap<>();
        for (String login : logins) {
            Gaps gaps = gapApi.exportGaps(event.getStart(), event.getEnd(), login);
            result.putAll(gaps.getPersons());
        }
        return result;
    }

    private Integer getOfficeId(Map<String, List<Gap>> gaps, Person person, EventDate event) {
        Set<String> personGaps = gaps.getOrDefault(person.getLogin(), Collections.emptyList()).stream()
                .filter(gap -> EventGapAnalyzer.isIn(gap, event))
                .map(Gap::getWorkflow)
                .collect(Collectors.toSet());
        if (isAbsent(TRUE_ABSENCE, personGaps)) {
            return ABSENT;
        } else if (isAbsent(TRIP, personGaps)) {
            return IN_TRIP;
        } else {
            return person.getLocation().getOffice().getId();
        }
    }

    private boolean isAbsent(Set<String> absences, Set<String> gapTypes) {
        return gapTypes.stream()
                .anyMatch(absences::contains);
    }

    public Set<Integer> officesToAdd(Event event, Contexts.Scan context, Person requester) {
        Set<Integer> officeIds;
        if (context.isAuto()) {
            officeIds = new HashSet<>(split(event, requester).keySet());
        } else {
            officeIds = context.getOfficeIds();
        }

        for (Resource.Info info : event.getResources()) {
            officeIds.remove(info.getOfficeId());
        }

        return officeIds;
    }
}
