package ru.yandex.qe.mail.meetings.utils;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.TimeZone;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import ru.yandex.qe.mail.meetings.services.calendar.dto.Repetition;

/**
 * @author Sergey Galyamichev
 */
public class RepetitionUtils {
    private static final String TIME_FORMAT = "yyyy-MM-dd'T'HH:mm:ss";

    private static final ThreadLocal<DateFormat> timeConverterHolder =
            ThreadLocal.withInitial(() -> {
                SimpleDateFormat sdf = new SimpleDateFormat(TIME_FORMAT);
                sdf.setTimeZone(TimeZone.getTimeZone("GMT"));
                return sdf;
            });

    private static final Map<String, Integer> DAYS = Map.of(
            "sun", Calendar.SUNDAY,
            "mon", Calendar.MONDAY,
            "tue", Calendar.TUESDAY,
            "wed", Calendar.WEDNESDAY,
            "thu", Calendar.THURSDAY,
            "fri", Calendar.FRIDAY,
            "sat", Calendar.SATURDAY
    );

    private static final Map<Integer, String> RDAYS = DAYS.entrySet().stream().collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey));

    public static Date getNextRepetition(Date start, Repetition repetition) {
        Calendar c = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
        c.setTime(start);
        if (repetition.getType() == Repetition.Type.DAILY) {
            c.add(Calendar.DAY_OF_MONTH, repetition.getEach());
        } else if (repetition.getType() == Repetition.Type.WEEKLY) {
            int[] days = Arrays.stream(repetition.getWeeklyDays().split(","))
                    .mapToInt(DAYS::get)
                    .sorted()
                    .toArray();
            int dayOfWeek = c.get(Calendar.DAY_OF_WEEK);
            for (int i = 0; i < days.length; i++) {
                if (dayOfWeek == days[i]) {
                    if (i + 1 == days.length) {
                        c.add(Calendar.DAY_OF_MONTH, 7 * repetition.getEach() - dayOfWeek + days[0]);
                    } else {
                        c.add(Calendar.DAY_OF_MONTH, days[i + 1] - dayOfWeek);
                    }
                    break;
                }
            }
        } else if (repetition.getType() == Repetition.Type.MONTHLY_NUMBER) {
            c.add(Calendar.MONTH, repetition.getEach());
        } else if (repetition.getType() == Repetition.Type.MONTHLY_DAY_WEEKNO) {
            final int expectedMonth = (c.get(Calendar.MONTH) + repetition.getEach()) % 12;
            while (c.get(Calendar.MONTH) != expectedMonth) {
                c.add(Calendar.WEEK_OF_YEAR, 1);
            }
        } else if (repetition.getType() == Repetition.Type.YEARLY) {
            c.add(Calendar.YEAR, repetition.getEach());
        }
        Date result = c.getTime();
        DateRange.toStartOfADay(c);
        if (repetition.getDueDate() != null && c.getTimeInMillis() > repetition.getDueDate().getTime()) {
            throw new NoSuchElementException("next occurrence after " + repetition.getDueDate());
        }
        return result;
    }

    public static String fromTime(Date date) {
        return timeConverterHolder.get().format(date);
    }

    public static Date toTime(String string) throws Exception {
        return timeConverterHolder.get().parse(string);
    }

    public static String next(String instance, Repetition repetition) throws Exception {
        return fromTime(getNextRepetition(toTime(instance), repetition));
    }

    public static Repetition repeatWeeklyOn(@Nonnull Set<Integer> days, int each) {
        var dayString = days.stream().map(RDAYS::get).collect(Collectors.joining(","));
        return repeatWeeklyOn(dayString, each);
    }

    public static Repetition repeatWeeklyOn(String days, int each) {
        Repetition repetition = buildRepetition(Repetition.Type.WEEKLY, each);
        repetition.setWeeklyDays(days);
        return repetition;
    }

    public static Repetition buildRepetition(Repetition.Type type, int each) {
        Repetition repetition = new Repetition();
        repetition.setType(type);
        repetition.setEach(each);
        repetition.setWeeklyDays("");
        return repetition;
    }
}
