package ru.yandex.qe.mail.meetings.ws;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.ws.rs.NotFoundException;

import com.codahale.metrics.MetricRegistry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;

import ru.yandex.qe.mail.meetings.api.resource.CalendarActions;
import ru.yandex.qe.mail.meetings.api.resource.dto.ActionType;
import ru.yandex.qe.mail.meetings.api.resource.dto.AddResourceRequest;
import ru.yandex.qe.mail.meetings.api.resource.dto.CalendarAction;
import ru.yandex.qe.mail.meetings.api.resource.dto.MergeEventRequest;
import ru.yandex.qe.mail.meetings.api.resource.dto.Status;
import ru.yandex.qe.mail.meetings.api.resource.dto.SwapResourceRequest;
import ru.yandex.qe.mail.meetings.cron.AbstractMessageBuilder;
import ru.yandex.qe.mail.meetings.cron.actions.ActionsRunner;
import ru.yandex.qe.mail.meetings.cron.actions.MergeResource;
import ru.yandex.qe.mail.meetings.cron.actions.ResourceSwapper;
import ru.yandex.qe.mail.meetings.cron.actions.email.SwapRequestMessageBuilder;
import ru.yandex.qe.mail.meetings.cron.scanner.FailureMessageBuilder;
import ru.yandex.qe.mail.meetings.rooms.dao.SentEmailsDao;
import ru.yandex.qe.mail.meetings.services.calendar.CalendarUpdate;
import ru.yandex.qe.mail.meetings.services.calendar.CalendarWeb;
import ru.yandex.qe.mail.meetings.services.calendar.dto.faults.CalendarException;
import ru.yandex.qe.mail.meetings.services.staff.StaffClient;
import ru.yandex.qe.mail.meetings.utils.FormConverters;
import ru.yandex.qe.mail.meetings.utils.HtmlBuilder;

/**
 * @author Sergey Galyamichev
 */
@Service("calendarActions")
public class CalendarActionsImpl implements CalendarActions {
    private static final String NO_ACCESS_TEXT = "Нет доступа к действию!";
    public static final String PERFORMED_TEXT = "Все уже сделано!)";

    private static final Logger LOG = LoggerFactory.getLogger(CalendarActionsImpl.class);

    private final SentEmailsDao sentEmailsDao;
    private final CalendarWeb calendarWeb;
    private final CalendarUpdate calendarUpdate;
    private final ActionsRunner runner;
    private final HtmlBuilder htmlBuilder;
    private final MetricRegistry metricRegistry;
    private final JavaMailSender mailSender;
    private final SearchMessageBuilder searchMessageBuilder;
    private final FailureMessageBuilder failureMessageBuilder;
    private final SwapRequestMessageBuilder swapRequestMessageBuilder;
    private final StaffClient staffClient;


    public CalendarActionsImpl(SentEmailsDao sentEmailsDao,
                               CalendarWeb calendarWeb,
                               CalendarUpdate calendarUpdate,
                               ActionsRunner runner,
                               HtmlBuilder htmlBuilder,
                               MetricRegistry metricRegistry,
                               JavaMailSender mailSender,
                               SearchMessageBuilder searchMessageBuilder,
                               FailureMessageBuilder failureMessageBuilder,
                               SwapRequestMessageBuilder swapRequestMessageBuilder,
                               StaffClient staffClient) {
        this.sentEmailsDao = sentEmailsDao;
        this.calendarWeb = calendarWeb;
        this.calendarUpdate = calendarUpdate;
        this.runner = runner;
        this.htmlBuilder = htmlBuilder;
        this.metricRegistry = metricRegistry;
        this.mailSender = mailSender;
        this.searchMessageBuilder = searchMessageBuilder;
        this.failureMessageBuilder = failureMessageBuilder;
        this.swapRequestMessageBuilder = swapRequestMessageBuilder;
        this.staffClient = staffClient;
    }

    @Override
    public List<CalendarAction> pendingActions() {
        return sentEmailsDao.getActions(Status.PENDING);
    }

    @Override
    public String getStatus(String actionId) {
        try {
            return sentEmailsDao.getAction(actionId).getStatus().name();
        } catch (EmptyResultDataAccessException e) {
            throw new NotFoundException(String.format("Action %s not found ", actionId));
        }
    }

    @Override
    public String getAddResourceActions(String login) {
//        String login = SecurityContextHolder.getContext().getAuthentication().getPrincipal().toString();
        List<CalendarAction> actions = sentEmailsDao.getActions(ActionType.ADD_RESOURCE, Status.ACCEPTED).stream()
                .filter(a -> login == null || AbstractMessageBuilder.getLogin(a.getEmail()).equals(login))
                .collect(Collectors.toList());
        return HtmlBuilder.toPretty(actions);
    }

    @Override
    public String findResource(int eventId, String officeId, int notifyBefore, Boolean isAuto, String filter, String login, String eventDate) {
        LOG.info("findResource(eventId = {}, officeId = {}, notifyBefore = {})", eventId, officeId, notifyBefore);
        AddResourceRequest request = new AddResourceRequest();
        request.setTtl("" + notifyBefore);
        request.setFilter(filter);
        request.setOffice(officeId);
        boolean auto = Boolean.TRUE.equals(isAuto);
        request.setManualOffice(auto ? "False" : "True");
        if (!auto) {
            String officeNames = Arrays.stream(officeId.split(","))
                    .map(Integer::valueOf)
                    .map(FormConverters.OFFICES_IDS::get)
                    .collect(Collectors.joining(","));
            request.setOffice(officeNames);
        }
        request.setCalendarUrl("https://" + EventResourceDescriptor.CALENDAR_YANDEX_TEAM_RU + "/event/" + eventId +
                (eventDate == null ? "" : "?event_date=" + eventDate));
        return findResource(login, request);
    }

    @Override
    public String findResource(@Nonnull String organizer, @Nonnull AddResourceRequest request) {
        LOG.info("findResource(organizer = {}, request = {})", organizer, request);
        SearchAction searchAction = new SearchAction(calendarWeb, staffClient);
        CalendarAction action = searchAction.prepare(request, organizer);
        if (searchAction.isValid()) {
            List<CalendarAction> runningActions = sentEmailsDao.getActions(action.getEventId(), ActionType.ADD_RESOURCE)
                    .stream().filter(a -> a.getStatus() == Status.ACCEPTED).collect(Collectors.toList());

            runningActions.forEach(a -> sentEmailsDao.updateAction(a.getActionId(), Status.DECLINED));

            sentEmailsDao.insertActions(Collections.singleton(action));
            mailSender.send(mimeMessage -> searchMessageBuilder.prepareMessage(mimeMessage, action, searchAction.getEvent()));
            metricRegistry.counter(MetricRegistry.name(CalendarActions.class, action.getType().name().toLowerCase(),
                    action.getStatus().name().toLowerCase())).inc();

            return action.getActionId();
        } else {
            mailSender.send(mimeMessage -> failureMessageBuilder.prepareMessage(mimeMessage, action, "Validation error!"));
            LOG.error("Validation result {}", searchAction.getResult());
            return "Error";
        }
    }

    @Override
    public String swapResource(@Nonnull String requester, @Nonnull SwapResourceRequest request) {
        LOG.info("swapResource(requester = {}, request = {})", requester, request);
        ResourceSwapper swapper = new ResourceSwapper(calendarWeb, staffClient);
        CalendarAction action = swapper.prepare(request, requester);
        action.setStatus(swapper.tryExecute(action));
        sentEmailsDao.insertActions(Collections.singleton(action));
        if (action.getStatus() == Status.PENDING) {
            mailSender.send(mimeMessage -> swapRequestMessageBuilder.prepareMessage(mimeMessage, action, swapper, request.getDescription()));
        }
        metricRegistry.counter(MetricRegistry.name(CalendarActions.class, action.getType().name().toLowerCase(),
                action.getStatus().name().toLowerCase())).inc();
        return action.getActionId();
    }

    @Override
    public void mergeEvent(@Nonnull String organizer, @Nonnull MergeEventRequest request) {
        LOG.info("mergeEvent(organizer = {}, request = {})", organizer, request);
        MergeResource mergeAction = new MergeResource(calendarWeb, staffClient, organizer);
        mergeAction.execute(calendarUpdate, request);
    }

    @Override
    public String cancelAction(String actionId) {
        LOG.info("Cancel action is = {}", actionId);
        CalendarAction action = sentEmailsDao.getAction(actionId);
        if (isOwner(action)) {
            return sentEmailsDao.updateAction(actionId, Status.DECLINED) ? PERFORMED_TEXT :
                    "Unable to cancel request";
        } else {
            return NO_ACCESS_TEXT;
        }
    }

    @Override
    public String executeAction(String actionId) {
        LOG.info("executeAction(actionId = {})", actionId);
        CalendarAction action = sentEmailsDao.getAction(actionId);
        LOG.info("executeAction(action = {})", action);
        if (isOwner(action)) {
            if (doAction(action) == Status.PERFORMED) {
                return htmlBuilder.buildResponse(Collections.singletonList(action));
            }
            return PERFORMED_TEXT;
        } else {
            return NO_ACCESS_TEXT;
        }
    }

    private Status doAction(CalendarAction action) {
        Status status = Status.PERFORMED;
        try {
            status = runner.perform(action);
            metricRegistry.counter(MetricRegistry.name(CalendarActions.class, action.getType().name().toLowerCase(),
                    status.name().toLowerCase())).inc();
        } catch (CalendarException e) {
            LOG.error("Action hasn't been performed due to the exception:", e);
        }
        sentEmailsDao.updateAction(action.getActionId(), status);
        return status;
    }

    @Override
    public String cancelGroup(String groupId) {
        LOG.info("cancelGroup(groupId = {})", groupId);
        List<CalendarAction> actions = sentEmailsDao.getActions(groupId);
        for (CalendarAction action : actions) {
            if (isOwner(action)) {
                sentEmailsDao.updateAction(action.getActionId(), Status.DECLINED);
            }
        }
        return PERFORMED_TEXT;
    }

    @Override
    public String executeGroup(String groupId) {
        LOG.info("executeGroup(groupId = {})", groupId);
        List<CalendarAction> actions = sentEmailsDao.getActions(groupId);
        List<CalendarAction> performed = new ArrayList<>();
        for (CalendarAction action : actions) {
            if (isOwner(action) && doAction(action) == Status.PERFORMED) {
                performed.add(action);
            }
        }
        return performed.isEmpty() ? PERFORMED_TEXT : htmlBuilder.buildResponse(performed);
    }

    private boolean isOwner(CalendarAction action) {
        return true;
//        String login = SecurityContextHolder.getContext().getAuthentication().getPrincipal().toString();
//        return AbstractMessageBuilder.getLogin(action.getEmail()).equals(login);
    }

    @Override
    public String ping() {
        LOG.info("ping");
        LOG.info("login {}", SecurityContextHolder.getContext().getAuthentication().getPrincipal());
        return "pong";
    }
}
