package ru.yandex.qe.mail.meetings.ws;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.qe.mail.meetings.analisys.DismissedUserAnalyzer;
import ru.yandex.qe.mail.meetings.analisys.EventGapAnalyzer;
import ru.yandex.qe.mail.meetings.analisys.EventGapsInspectionsBuilder;
import ru.yandex.qe.mail.meetings.analisys.InspectionsResult;
import ru.yandex.qe.mail.meetings.analisys.UserCheck;
import ru.yandex.qe.mail.meetings.blamer.DeclineEvent;
import ru.yandex.qe.mail.meetings.domain.EventExportObject;
import ru.yandex.qe.mail.meetings.domain.EventRecord;
import ru.yandex.qe.mail.meetings.services.calendar.CalendarWeb;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Decision;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Event;
import ru.yandex.qe.mail.meetings.services.calendar.dto.EventDate;
import ru.yandex.qe.mail.meetings.services.calendar.dto.EventUser;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Events;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Office;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Offices;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Resource;
import ru.yandex.qe.mail.meetings.services.calendar.dto.User;
import ru.yandex.qe.mail.meetings.services.calendar.dto.faults.CalendarException;
import ru.yandex.qe.mail.meetings.services.calendar.dto.faults.Error;
import ru.yandex.qe.mail.meetings.services.gaps.dto.Gap;
import ru.yandex.qe.mail.meetings.utils.CalendarHelper;

@Component
public class CalendarFacade {
    private static final Logger LOG = LoggerFactory.getLogger(CalendarFacade.class);

    private static final String TIME_FORMAT = "yyyy-MM-dd'T'HH:mm:ss";
    private static final String ORGANIZER_UNKNOWN = "Organizer Unknown";
    private static final String EVENT_DELETED = "Event Deleted";
    private static final String PERMISSION_DENIED = "Permission Denied";
    private static final String UNKNOWN_ERROR = "Unknown Error";

    private static ThreadLocal<DateFormat> timeConverterHolder =
            ThreadLocal.withInitial(() -> new SimpleDateFormat(TIME_FORMAT));

    @Value("${declines.notify.user:g-s-v@yandex-team.ru}")
    private String notifyUser;

    private final CalendarHelper calendar;
    private final CalendarWeb calendarWeb;

    public CalendarFacade(CalendarHelper calendar, CalendarWeb calendarWeb) {
        this.calendar = calendar;
        this.calendarWeb = calendarWeb;
    }

    private List<Office> getEventsTree(Date from, Date to) {
        return calendarWeb.getOffices().getOffices().stream()
                .map(Office::getId)
                .map(id -> calendarWeb.getResourceSchedule(List.of(id), from, to))
                .flatMap(os -> os.getOffices().stream())
                .collect(Collectors.toList());
    }

   private <T> List<T> iterateEvents(Date from, Date to, Function<EventObject, T> mapper) {
        List<T> result = new ArrayList<>();
        for (Office office : calendarWeb.getOffices().getOffices()) {
            Offices officeSchedule = calendarWeb.getResourceSchedule(List.of(office.getId()), from, to);
            for (Office schedule : officeSchedule.getOffices()) {
                List<Resource> resources = schedule.getResources();
                for (Resource resource : resources) {
                    for (EventDate eventDate : resource.getEvents()) {
                        if (eventDate.isEvent()) {
                            T event = mapper.apply(new EventObject(office, resource, eventDate));
                            if (event != null) {
                                result.add(event);
                            }
                        }
                    }
                }
            }
        }
        return result;
    }

    public List<EventExportObject> getEvents(Date from, Date to) {
        return iterateEvents(from, to, this::converter);
    }

    private EventExportObject converter(EventObject event) {
        EventDate eventDate = event.getEvent();
        int size = 0;
        try {
            size = calendarWeb.getEvent(eventDate.getEventId()).getAttendees().size();
        } catch (Exception e) {
            LOG.warn("Unable to get event info due to the exception", e);
        }
        long duration = TimeUnit.MILLISECONDS.toMinutes(eventDate.getEnd().getTime() - eventDate.getStart().getTime());
        return new EventExportObject(eventDate.getEventId(),
                event.getOffice().getId(),
                event.getResource().getResourceInfo().getId(),
                timeConverterHolder.get().format(eventDate.getStart()),
                timeConverterHolder.get().format(eventDate.getEnd()),
                duration,
                size) ;
    }

    public List<InspectionsResult> checkEvents(Date from, Date to, UserCheck userCheck, Map<String, List<Gap>> gaps) {
        EventGapAnalyzer<InspectionsResult> eventGapAnalyzer = new EventGapAnalyzer<>(calendar, calendarWeb, userCheck, gaps, EventGapsInspectionsBuilder::new);
        return iterateEvents(from, to, eventGapAnalyzer::check);
    }

    public List<Pair<Event, InspectionsResult>> checkEvents(Date from, Date to, UserCheck userCheck) {
        DismissedUserAnalyzer dismissedUserAnalyzer = new DismissedUserAnalyzer(calendarWeb, userCheck);
        return iterateEvents(from, to, dismissedUserAnalyzer::check);
    }

    public List<Pair<Event, List<User>>> findDismissedAttendees(Date from, Date to, UserCheck userCheck) {
        DismissedUserAnalyzer dismissedUserAnalyzer = new DismissedUserAnalyzer(calendarWeb, userCheck);
        return iterateEvents(from, to, dismissedUserAnalyzer::findAttendees);
    }

    public <T> Map<Integer, Event> getBrief(List<T> events, Function<T, Integer> idMapper) {
        String allIds = events.stream()
                .map(idMapper)
                .filter(Objects::nonNull)
                .map(Objects::toString)
                .collect(Collectors.joining(","));
        Events eventsBrief = calendarWeb.getEventsBrief(allIds, true);
        return eventsBrief.getEvents().stream()
                .collect(Collectors.toMap(Event::getEventId, e -> e));
    }

    public List<EventRecord> getEventRecords(Date from, Date to) {
        DateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        List<EventRecord> result = new ArrayList<>();
        for (Office office : calendarWeb.getOffices().getOffices()) {
            Offices officeSchedule = calendarWeb.getResourceSchedule(List.of(office.getId()), from, to);
            for (Office schedule : officeSchedule.getOffices()) {
                List<Resource> resources = schedule.getResources();
                for (Resource resource : resources) {
                    Map<Integer, Event> eventMap = getBrief(resource.getEvents(), EventDate::getEventId);
                    for (EventDate eventDate : resource.getEvents()) {
                        try {
                            Event event = eventMap.get(eventDate.getEventId());
                            result.add(new EventRecord(eventDate.getEventId(),
                                    office.getId(),
                                    resource.getResourceInfo().getId(),
                                    format.format(eventDate.getStart()),
                                    format.format(eventDate.getEnd()),
                                    event.getOrganizer(),
                                    getAttendees(event),
                                    event.getAttendees()
                            ));
                        } catch (Exception e) {
                            LOG.warn("error on event: " + eventDate, e);
                        }
                    }
                }
            }
        }
        return result;
    }

    private long getAttendees(Event event) {
        return event.getAttendees().stream().filter(eventUser -> eventUser.getDecision() != Decision.NO).count();
    }

    public List<Resource.Info> getResources() {
        List<Resource.Info> result = new ArrayList<>();
        for (Office office : calendarWeb.getOffices().getOffices()) {
            Offices offices = calendarWeb.getResourceSchedule(office.getId(), new Date());
            for (Office officesOffice : offices.getOffices()) {
                for (Resource resource : officesOffice.getResources()) {
                    Resource.Info resourceInfo = calendarWeb.getResourceInfo(resource.getResourceInfo().getEmail());
                    if (resourceInfo != null) {
                        Resource.Info info = resource.getResourceInfo();
                        Resource.Info updated = new Resource.Info();
                        updated.setId(info.getId());
                        updated.setOfficeId(resourceInfo.getOfficeId());
                        updated.setName(resourceInfo.getName());
                        updated.setEmail(resourceInfo.getEmail());
                        updated.setPhone(resourceInfo.getPhone());
                        updated.setVideo(resourceInfo.getVideo());
                        updated.setDescription(resourceInfo.getDescription());
                        updated.setSeats(resourceInfo.getSeats());
                        updated.setCapacity(resourceInfo.getCapacity());
                        updated.setVoiceConferencing(resourceInfo.isVoiceConferencing());
                        updated.setProjector(resourceInfo.getProjector());
                        updated.setLcdPanel(resourceInfo.getLcdPanel());
                        updated.setMarkerBoard(resourceInfo.isMarkerBoard());
                        updated.setDesk(resourceInfo.isDesk());
                        updated.setActive(resourceInfo.isActive());
                        updated.setResourceType(resourceInfo.getResourceType());
                        updated.setHasPhone(info.isHasPhone());
                        updated.setHasVideo(info.isHasVideo());
                        updated.setFloor(resourceInfo.getFloor());
                        updated.setGroup(info.getGroup());
                        updated.setCanBook(info.getCanBook());
                        updated.setProtection(info.getProtection());
                        result.add(updated);
                    }
                }
            }
        }
        return result;
    }

    public void patchEvent(DeclineEvent declinedEvent) {
        try {
            Event event = calendarWeb.getEvent(declinedEvent.getEventId());
            EventUser organizer = event.getOrganizer();
            if (organizer != null) {
                setOrganizer(declinedEvent, organizer.getEmail(), organizer.getName(), organizer.getLogin());
            } else {
                setOrganizer(declinedEvent, notifyUser, ORGANIZER_UNKNOWN, ORGANIZER_UNKNOWN);
                LOG.warn("Can't find organizer for {} resource {}", declinedEvent.getEventId(), declinedEvent.getResourceEmail());
            }
        } catch (CalendarException ce) {
            if (Error.EVENT_NOT_FOUND.equals(ce.getName())) {
                setOrganizer(declinedEvent, notifyUser, EVENT_DELETED, EVENT_DELETED);
            } else if (Error.NO_PERMISSIONS_FOR_EVENT_ACTION.equals(ce.getName())) {
                setOrganizer(declinedEvent, notifyUser, PERMISSION_DENIED, PERMISSION_DENIED);
            }
            LOG.warn("Event {} skipped in {} due {}", declinedEvent.getEventId(), declinedEvent.getResourceEmail(), ce.getMessage());
        } catch (Exception e) {
            setOrganizer(declinedEvent, notifyUser, UNKNOWN_ERROR, UNKNOWN_ERROR);
            LOG.warn("Event {} skipped in {}", declinedEvent.getEventId(), declinedEvent.getResourceEmail(), e);
        }
    }

    private void setOrganizer(DeclineEvent declinedEvent, String email, String name, String login) {
        declinedEvent.setEmail(email);
        declinedEvent.setName(name);
        declinedEvent.setLogin(login);
    }

    public List<String> findOrganizers(Set<String> rooms, Date from, Date to) {
        return iterateEvents(from,to, e -> getOrganizer(rooms, e));
    }

    private String getOrganizer(Set<String> rooms, EventObject eventObject) {
        if (rooms.contains(eventObject.getResource().getResourceInfo().getEmail())) {
            Event event = calendarWeb.getEvent(eventObject.getEvent().getEventId());
            LOG.info("Event matched {}", eventObject);
            return event.getOrganizer() != null ?
                    event.getOrganizer().getEmail() :
                    "no organizer has been found for " + "https://calendar.yandex-team.ru/event/" + eventObject.getEvent().getEventId();
        } else {
            return null;
        }
    }
}
