package ru.yandex.qe.mail.meetings.ws;

import java.io.IOException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import javax.ws.rs.BadRequestException;

import org.apache.http.NameValuePair;
import org.apache.http.client.utils.URLEncodedUtils;
import org.apache.http.message.BasicNameValuePair;

import ru.yandex.qe.mail.meetings.services.calendar.CalendarWeb;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Event;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Events;
import ru.yandex.qe.mail.meetings.services.staff.StaffClient;
import ru.yandex.qe.mail.meetings.services.staff.dto.Person;

/**
 * @author Sergey Galyamichev
 */
public class EventResourceDescriptor {
    public static final String CALENDAR_YANDEX_TEAM_RU = "calendar.yandex-team.ru";
    private final int eventId;
    private final String login;
    private final String instanceTs;

    public EventResourceDescriptor(int eventId,
                                   String login,
                                   String instanceTs) {
        this.eventId = eventId;
        this.login = login;
        this.instanceTs = instanceTs;
    }

    public int getEventId() {
        return eventId;
    }

    public String getLogin() {
        return login;
    }

    public String getInstanceTs() {
        return instanceTs;
    }

    public static EventResourceDescriptor fromRequest(CalendarWeb calendarWeb,
                                                      StaffClient staffClient,
                                                      String organizer,
                                                      String calendarUrl) throws IOException {
        URL url = new URL(calendarUrl);
        if (!CALENDAR_YANDEX_TEAM_RU.equals(url.getHost())) {
            throw new BadRequestException("Calendar URL is wrong!");
        }
        List<NameValuePair> pairs = new ArrayList<>();
        if (url.getQuery() == null) {
            pairs.add(new BasicNameValuePair("event_id", url.getPath().substring("/event/".length())));
        } else {
            pairs.addAll(URLEncodedUtils.parse(url.getQuery(), StandardCharsets.UTF_8));
        }
        Optional<String> eventIdParam = getField(pairs, "event_id");
        if (eventIdParam.isPresent()) {
            return fromEventId(calendarWeb, eventIdParam.get(), organizer);
        }
        String eventDate = getField(pairs, "event_date")
                .orElseThrow();
        String applyToFuture = getField(pairs, "applyToFuture")
                .orElse("0");
        int index = url.getPath().lastIndexOf('/');
        if (index < 0) {
            throw new BadRequestException("Can't find event id");
        }
        try {
            int eventId = Integer.parseInt(url.getPath().substring(index + 1));
            String date = eventDate;
            Person person = staffClient.getByLogin(organizer);
            if (person != null) {
                Events events = calendarWeb.getEvents(person.getUid(), eventId, eventDate, eventDate);
                date = "1".equals(applyToFuture) ? null : events.getEvents().stream()
                        .findAny()
                        .map(Event::getInstanceStartTs)
                        .orElse(eventDate);
            }
            return new EventResourceDescriptor(eventId, organizer, date);
        } catch (NumberFormatException e) {
            throw new BadRequestException("Event is invalid: " + e.getMessage());
        }
    }

    private static EventResourceDescriptor fromEventId(CalendarWeb calendarWeb,
                                                       String eventIdParam,
                                                       String organizer) {
        try {
            int eventId = Integer.parseInt(eventIdParam);
            Event event = calendarWeb.getEvent(eventId);
            String instanceTs = event.getIsRecurrence() ? event.getInstanceStartTs() : null;
            return new EventResourceDescriptor(eventId, organizer, instanceTs);
        } catch (NumberFormatException e) {
            throw new BadRequestException("Event is invalid: " + e.getMessage());
        }
    }

    private static Optional<String> getField(List<NameValuePair> pairs, String fieldName) {
        return pairs.stream().filter(p -> fieldName.equals(p.getName()))
                .map(NameValuePair::getValue)
                .findFirst();
    }

    public Event findEvent(CalendarWeb calendarWeb, StaffClient staffClient, String date) {
        Event event = calendarWeb.getEvent(eventId);
        if (date != null && !date.isBlank()) {
            Person org = staffClient.getByLogin(event.getOrganizer().getLogin());
            event = calendarWeb.getEvents(org.getUid(), eventId, date, date).getEvents().stream()
                    .findAny()
                    .orElseThrow();
        }
        return event;
    }

    @Override
    public String toString() {
        return "EventResourceDescriptor{" +
                "eventId=" + eventId +
                ", login='" + login + '\'' +
                ", instanceTs='" + instanceTs + '\'' +
                '}';
    }
}


