package ru.yandex.qe.mail.meetings.ws;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.inject.Inject;

import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.DateTimeFormatterBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.qe.mail.meetings.analisys.InspectedItem;
import ru.yandex.qe.mail.meetings.analisys.InspectionsResult;
import ru.yandex.qe.mail.meetings.api.resource.ExportApiService;
import ru.yandex.qe.mail.meetings.blamer.DeclineEvent;
import ru.yandex.qe.mail.meetings.blamer.DeclinedEventsDao;
import ru.yandex.qe.mail.meetings.cron.dismissed.DismissedAttendeesEventsProvider;
import ru.yandex.qe.mail.meetings.cron.forecast.PossiblyDeclinedEventProvider;
import ru.yandex.qe.mail.meetings.domain.EventExportObject;
import ru.yandex.qe.mail.meetings.domain.EventRecord;
import ru.yandex.qe.mail.meetings.rooms.ResourceDao;
import ru.yandex.qe.mail.meetings.services.calendar.DateParameterConverterProvider;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Event;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Resource;
import ru.yandex.qe.mail.meetings.services.calendar.dto.User;
import ru.yandex.qe.mail.meetings.services.staff.StaffClient;
import ru.yandex.qe.mail.meetings.services.staff.dto.Person;
import ru.yandex.qe.mail.meetings.storage.CalendarStorage;
import ru.yandex.qe.mail.meetings.storage.s3.YTStorage;
import ru.yandex.qe.mail.meetings.utils.DateRange;

/**
 * @author selivanov 2019-05-24
 */
@Service("exportApiService")
public class ExportApiServiceImpl implements ExportApiService {
    private static final Logger LOG = LoggerFactory.getLogger(ExportApiServiceImpl.class);

    private static final DateTimeFormatter FORMATTER = new DateTimeFormatterBuilder()
            .append(DateTimeFormat.forPattern("yyyy-MM-dd"))
            .toFormatter();

    private static final DateTimeFormatter EVENTS_FORMATTER = new DateTimeFormatterBuilder()
            .append(DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss"))
            .toFormatter();
    private static final String CALENDAR_LINK_PATTERN = "https://calendar.yandex-team.ru/event/%d";

    @Inject
    private CalendarFacade calendarFacade;
    @Inject
    private YTStorage ytStorage;
    @Value("${calendar.yt.root.path}")
    private String ytPath;
    @Inject
    private PossiblyDeclinedEventProvider possiblyDeclinedEventProvider;
    @Inject
    private DeclinedEventsDao declinedEventsDao;
    @Inject
    private ResourceDao resourceDao;
    @Inject
    private StaffClient staffClient;
    @Inject
    private CalendarStorage calendarStorage;
    @Inject
    private TaskExecutor taskExecutor;
    @Inject
    private DismissedAttendeesEventsProvider dismissedAttendeesEventsProvider;


    @Override
    public boolean exportEvents(@Nonnull Date from, @Nonnull Date to) {
        DateTime dateTime = DateTime.now();
        LOG.info("from = [{}}], to = [{}}]", from, to);
        List<EventRecord> eventRecords = calendarFacade.getEventRecords(from, to);
        eventRecords.sort(Comparator.comparing(EventRecord::getEventId));
        String path = getTableName(DateParameterConverterProvider.CONVERTER.toString(from),
                DateParameterConverterProvider.CONVERTER.toString(to), dateTime);
        boolean result = calendarStorage.storeEvents(path, eventRecords);
        LOG.info("export {} done to {}", eventRecords.size(), path);
        return result;
    }

    private String getTableName(@Nonnull String from, @Nonnull String to, DateTime dateTime) {
        String path;
        if (from.equals(to)) {
            String suffix = "";
            int attempt = 0;
            do {
                path = ytPath + "/schedule/" + from + suffix;
                attempt++;
                suffix = "-" + attempt;
            } while (ytStorage.exists(path));
        } else {
            path = String.format("%s/%s/%s", ytPath, dateTime.toString(FORMATTER), dateTime.toString(EVENTS_FORMATTER));
        }
        return path;
    }

    @Override
    public boolean exportResources() {
        List<Resource.Info> values = calendarFacade.getResources();
        values.sort(Comparator.comparing(Resource.Info::getId)
                .thenComparing(Resource.Info::getOfficeId));
        String path = ytPath + "/resources/" + DateTime.now().toString(FORMATTER);
        boolean result = calendarStorage.storeResources(path, values);
        LOG.info("export {} done to {}", values.size(), path);
        return result;
    }

    @Override
    public boolean exportDeclines(@Nonnull Date date) {
        List<DeclineEvent> events = declinedEventsDao.getDeclineEvent(date);
        LOG.info("Found {} declined events", events.size());
        events.forEach(calendarFacade::patchEvent);
        String path = declinedEventsDao.getDeclinedEventLog(date);
        return calendarStorage.storeDeclines(path, events);
    }

    @Override
    public List<String> getTopDeclines(@Nonnull Date from, @Nonnull Date to, int threshold) {
        return declinedEventsDao.getDeclineEvent(DateRange.range(from, to), threshold).stream()
                .filter(event -> event.getName() != null)
                .map(event -> event.getName() + " " + event.getCount())
                .collect(Collectors.toList());
    }

    @Override
    public List<String> getDeclinesForUser(@Nonnull Date from, @Nonnull Date to, @Nonnull String email) {
        return declinedEventsDao.getDeclineEvent(DateRange.range(from, to), email).stream()
                .map(event -> String.format(CALENDAR_LINK_PATTERN, event.getEventId()))
                .collect(Collectors.toList());
    }

    @Override
    public List<String> checkDismissed(@Nonnull Date date) {
        Map<User, Map<Event, List<User>>> events = dismissedAttendeesEventsProvider.findDismissed(date, date);
        List<String> result = new ArrayList<>();
        for (Map.Entry<User, Map<Event, List<User>>> user : events.entrySet()) {
            StringBuilder resolution = new StringBuilder(user.getKey().getLogin())
                    .append(" has meeting with dismissed employees: ");
            for (Map.Entry<Event, List<User>> event : user.getValue().entrySet()) {
                resolution.append(event.getKey().getEventId()).append(" on ")
                        .append(event.getKey().getStart())
                        .append(" with ").append(event.getValue().stream()
                        .map(User::getLogin)
                        .collect(Collectors.joining(", ")))
                        .append(" ");
            }
            result.add(resolution.toString());
        }
        return result;
    }

    @Override
    public List<String> getRestrictedResources() {
        List<RestrictedResource> resources = resourceDao.getResources();
        Map<String, List<Person>> personMap = resources.stream().map(RestrictedResource::getUids)
                .filter(s -> s != null && !s.isEmpty())
                .collect(Collectors.toMap(Function.identity(), this::getPersons, (o, n) -> n));
        return resources.stream()
                .filter(r -> "t".equals(r.getExistsOnStaff()))
                .filter(r -> "t".equals(r.getrIsActive()))
                .sorted(Comparator.comparing(o -> Integer.valueOf(o.getCenterId())))
                .map(r -> toString(r, personMap.get(r.getUids())))
                .collect(Collectors.toList());
    }

    private String toString(RestrictedResource resource, List<Person> people) {
        String responsible = people != null ? people.stream()
                .map(Person::getLogin)
                .collect(Collectors.joining(", ")) : "";
        return String.format(
                "|| %s| %s| %s| %s| %s| %s| %s| %s||",
                resource.getName(),
                resource.getId(),
                resource.getCenterId(),
                resource.getCityNameEn(),
                responsible,
                resource.getProtectionMessage(), resource.getUrls(),
                !resource.getDisplayLastPingTs().isEmpty());
    }

    private List<Person> getPersons(String uids) {
        return Arrays.stream(uids.substring(1, uids.length() - 1).split(","))
                .map(staffClient::getByUid)
                .collect(Collectors.toList());
    }

    @Override
    public List<String> checkResourcesDeclines(@Nonnull Date date) {
        InspectionsResult result = possiblyDeclinedEventProvider.checkResourcesDeclines(date);
        return result.getAll().stream()
                .map(InspectedItem::toString)
                .collect(Collectors.toList());
    }

    @Override
    public String exportPublicEvents(@Nonnull Date from, @Nonnull Date to, @Nonnull String path) {
        return taskExecutor.run(() -> {
            List<EventExportObject> events = calendarFacade.getEvents(from, to);
            events.sort(Comparator.comparing(EventExportObject::getOfficeId)
                    .thenComparing(EventExportObject::getResourceId)
                    .thenComparing(EventExportObject::getStartTs));
            calendarStorage.storePublicEvents(path, events);
        });
    }

    @Override
    public String getStatus(@Nonnull String id) {
        return taskExecutor.status(id);
    }

    @Override
    public Set<String> getOrganizers(@Nonnull Date from, @Nonnull Date to, @Nonnull String resources) {
        Set<String> rooms = Stream.of(resources.split(","))
                .map(String::trim)
                .collect(Collectors.toSet());
        return new HashSet<>(calendarFacade.findOrganizers(rooms, from, to));
    }
}
