package ru.yandex.qe.mail.meetings.ws;

import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nonnull;
import javax.inject.Inject;
import javax.ws.rs.Consumes;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;

import com.codahale.metrics.MetricRegistry;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.cxf.jaxrs.ext.multipart.Multipart;
import org.apache.cxf.jaxrs.ext.multipart.MultipartBody;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.web.bind.annotation.RequestBody;

import ru.yandex.qe.mail.meetings.api.resource.CalendarActions;
import ru.yandex.qe.mail.meetings.api.resource.dto.AddResourceRequest;
import ru.yandex.qe.mail.meetings.api.resource.dto.MergeEventRequest;
import ru.yandex.qe.mail.meetings.api.resource.dto.SwapResourceRequest;
import ru.yandex.qe.mail.meetings.services.calendar.CalendarWeb;
import ru.yandex.qe.mail.meetings.services.staff.StaffClient;
import ru.yandex.qe.mail.meetings.utils.FormConverters;
import ru.yandex.qe.mail.meetings.ws.handlers.BookingHandler;
import ru.yandex.qe.mail.meetings.ws.handlers.MeetingSynchronizerHandler;
import ru.yandex.qe.mail.meetings.ws.validation.BookingValidator;
import ru.yandex.qe.mail.meetings.ws.validation.CalendarUrlValidator;
import ru.yandex.qe.mail.meetings.ws.validation.MergeEventValidator;
import ru.yandex.qe.mail.meetings.ws.validation.SearchResourceValidator;
import ru.yandex.qe.mail.meetings.ws.validation.SwapRequestValidator;
import ru.yandex.qe.mail.meetings.ws.validation.ValidationRequest;
import ru.yandex.qe.mail.meetings.ws.validation.ValidationResult;
import ru.yandex.qe.mail.meetings.ws.validation.Validator;

import static ru.yandex.qe.mail.meetings.api.resource.ExportApiService.APPLICATION_JSON_WITH_UTF;

@Service("formProcessingService")
@Produces(APPLICATION_JSON_WITH_UTF)
@Path("/form")
public class FormProcessor {
    private static final Logger LOG = LoggerFactory.getLogger(FormProcessor.class);
    private static final ObjectMapper MAPPER = new ObjectMapper();

    static {
        MAPPER.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    }

    @Nonnull
    private final CalendarActions calendarActions;

    @Nonnull
    private final BookingHandler booking;

    @Nonnull
    private final MetricRegistry metricRegistry;

    @Nonnull
    private final MeetingSynchronizerHandler meetingSynchronizerHandler;

    private final Map<ActionType, List<Validator>> validators;

    private final ThreadPoolExecutor executor;


    @Inject
    public FormProcessor(
            @Nonnull CalendarActions calendarActions,
            @Nonnull BookingHandler booking,
            @Nonnull CalendarWeb calendarWeb,
            @Nonnull MeetingSynchronizerHandler meetingSynchronizerHandler,
            @Nonnull StaffClient staffClient,
            @Nonnull MetricRegistry metricRegistry,
            @Value("${form.processor.incomig.queue.size:50}") int queueSize
    ) {
        this.calendarActions = calendarActions;
        this.booking = booking;
        this.metricRegistry = metricRegistry;
        this.meetingSynchronizerHandler = meetingSynchronizerHandler;
        this.executor = new ThreadPoolExecutor(1, 8, 60, TimeUnit.SECONDS, new ArrayBlockingQueue<>(queueSize));
        this.validators = Map.of(
                ActionType.SEARCH_CHANGE, List.of(new SearchResourceValidator(calendarWeb, staffClient)),
                ActionType.GROUP_ASSIGN, List.of(CalendarUrlValidator.with(calendarWeb, staffClient,
                        "calendar_url_sync")),
                ActionType.MERGE, List.of(new MergeEventValidator(calendarWeb, staffClient)),
                ActionType.BOOK, List.of(new BookingValidator()),
                ActionType.SWAP, List.of(new SwapRequestValidator(calendarWeb, staffClient))
        );
    }

    /**
     * метод обрабатывающий данные из объединенной формы
     * не выносим в интерфейс, т.к. во внешнем api ему делать нечего
     */
    @POST
    @Multipart
    @Consumes(MediaType.MULTIPART_FORM_DATA)
    @Path("process")
    public void processForm(@Nonnull MultipartBody mb, @Nonnull @QueryParam("login") String organizer) throws IOException {
        metricRegistry.counter(MetricRegistry.name(FormProcessor.class, "requests")).inc();

        // на стороне формы делать понятный запрос слишком сложно, поэтому выпарсим интересующие нас поля здесь
        // создадим json: {вопрос: ответ} и будем выпаршивать из него нужные объекты в зависимости от целевого метода

        Map<String, String> paramsMap = new HashMap<>();
        // не используем коллектор, т.к. там не разрешены null - value
        mb.getAllAttachments().stream()
                .map(attachment -> attachment.getObject(String.class))
                .filter(s -> !"content".equals(s)) // оставляем только json'ы с ответами
                .map(content -> {
                    try {
                        return MAPPER.readTree(content);
                    } catch (IOException e) {
                        LOG.error(String.format("json format error for: %s", content), e);
                        throw new RuntimeException(e);
                    }
                })
                .filter(tree -> Objects.nonNull(tree.get("value").textValue()))
                .forEach(tree -> {
                            LOG.info("part data: {}", tree);
                            paramsMap.put(
                                    tree.get("question").get("slug").textValue(),
                                    tree.get("value").textValue()
                            );
                        }
                );

        executeAction(organizer, paramsMap);
        LOG.info("task submitted");
    }

    @POST
    @Path("process-curl")
    public void processCurl(@Nonnull @QueryParam("login") String organizer, @RequestBody Map<String, String> paramsMap) {
        executeAction(organizer, paramsMap);
        LOG.info("task submitted");
    }

    private void executeAction(@QueryParam("login") @Nonnull String organizer, Map<String, String> paramsMap) {
        var paramsTree = MAPPER.convertValue(paramsMap, JsonNode.class);
        LOG.info("request data: {}", paramsMap.toString());

        ActionType actionType = Optional.ofNullable(paramsTree.get("action_type"))
                .map(JsonNode::textValue)
                .map(p -> FormConverters.convertOne(p, ActionType.FORM_MAPPING))
                .orElseThrow(() -> new IllegalArgumentException("field action_type is missing!"));

        metricRegistry.counter(MetricRegistry.name(FormProcessor.class, actionType.name())).inc();

        executor.execute(() -> chooseActionAndExecute(actionType, organizer, paramsMap, paramsTree));
    }

    @POST
    @Consumes(MediaType.APPLICATION_JSON)
    @Path("validate")
    public ValidationResult validate(ValidationRequest validationRequest) {
        LOG.debug("start validating request {}", validationRequest);
        String actionType = validationRequest.getAnswer("action_type");
        ActionType type = FormConverters.convertOne(actionType, ActionType.FORM_MAPPING);
        return validators.getOrDefault(type, Collections.emptyList()).stream()
                .map(validator -> validator.validate(validationRequest))
                .reduce(ValidationResult::merge)
                .orElse(ValidationResult.success());
    }

    private void chooseActionAndExecute(ActionType actionType, String organizer, Map<String, String> paramsMap,
                                        JsonNode paramsTree) {
        try {
            switch (actionType) {
                case BOOK:
                    LOG.info("start booking");
                    booking.process(organizer, paramsMap);
                    break;
                case SEARCH_CHANGE:
                    LOG.info("start searching");
                    AddResourceRequest resourceRequest = MAPPER.convertValue(paramsTree, AddResourceRequest.class);
                    calendarActions.findResource(organizer, resourceRequest);
                    break;
                case MERGE:
                    LOG.info("start merging");
                    MergeEventRequest mergeEventRequest = MAPPER.convertValue(paramsTree, MergeEventRequest.class);
                    calendarActions.mergeEvent(organizer, mergeEventRequest);
                    break;
                case GROUP_ASSIGN:
                    LOG.info("start group assigning");
                    meetingSynchronizerHandler.process(organizer, paramsMap);
                    break;
                case SWAP:
                    LOG.info("schedule swapping");
                    SwapResourceRequest swapResource = MAPPER.convertValue(paramsTree, SwapResourceRequest.class);
                    calendarActions.swapResource(organizer, swapResource);
                    break;
                default:
                    LOG.error("unexpected action type : {}", actionType);
                    throw new IllegalStateException("unexpected action type");
            }
            metricRegistry.counter(MetricRegistry.name(FormProcessor.class, actionType.name(), "success")).inc();
            LOG.info("request finished");
        } catch (Exception e) {
            metricRegistry.counter(MetricRegistry.name(FormProcessor.class, actionType.name(), "fatal")).inc();
            LOG.error("request failed", e);
        }
    }

    public enum ActionType {
        BOOK,
        SEARCH_CHANGE,
        MERGE,
        GROUP_ASSIGN,
        SWAP;

        static final Map<String, ActionType> FORM_MAPPING = Map.of(
                FormConverters.BOOK_LABEL, BOOK,
                FormConverters.SEARCH_LABEL, SEARCH_CHANGE,
                FormConverters.SEARCH_CHANGE_LABEL, SEARCH_CHANGE,
                FormConverters.MERGE_LABEL, MERGE,
                FormConverters.SWAP_LABEL, SWAP,
                FormConverters.SWAP_OR_MOVE_LABEL, SWAP,
                FormConverters.GROUP_ASSIGN_LABEL, GROUP_ASSIGN);
    }
}
