package ru.yandex.qe.mail.meetings.ws;

import java.util.Date;
import java.util.List;
import java.util.UUID;

import org.springframework.stereotype.Component;

import ru.yandex.qe.mail.meetings.api.resource.dto.ActionType;
import ru.yandex.qe.mail.meetings.api.resource.dto.AddResourceRequest;
import ru.yandex.qe.mail.meetings.api.resource.dto.CalendarAction;
import ru.yandex.qe.mail.meetings.api.resource.dto.Status;
import ru.yandex.qe.mail.meetings.cron.AbstractMessageBuilder;
import ru.yandex.qe.mail.meetings.cron.actions.Contexts;
import ru.yandex.qe.mail.meetings.services.calendar.CalendarWeb;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Event;
import ru.yandex.qe.mail.meetings.services.calendar.dto.faults.CalendarException;
import ru.yandex.qe.mail.meetings.services.staff.StaffClient;
import ru.yandex.qe.mail.meetings.utils.FormConverters;
import ru.yandex.qe.mail.meetings.ws.validation.ValidationResult;

import static ru.yandex.qe.mail.meetings.utils.DateUtils.minusMinutes;

/**
 * @author Sergey Galyamichev
 */
@Component
public class SearchAction {
    private final CalendarWeb calendarWeb;
    private final StaffClient staffClient;
    private Event event;
    private String organizer;
    private EventResourceDescriptor descriptor;
    private ValidationResult result;
    private int ttl;

    public SearchAction(CalendarWeb calendarWeb, StaffClient staffClient) {
        this.calendarWeb = calendarWeb;
        this.staffClient = staffClient;
    }

    public Event getEvent() {
        return event;
    }

    public ValidationResult getResult() {
        return result;
    }

    public ValidationResult validate(AddResourceRequest request) {
        ValidationResult result = ValidationResult.success();
        try {
            descriptor = EventResourceDescriptor.fromRequest(calendarWeb, staffClient, organizer, request.getCalendarUrl());
            event = descriptor.findEvent(calendarWeb, staffClient, null);
            Boolean change = FormConverters.convertYesNo(request.getChange());
            if (change) {
                List<Integer> office = FormConverters.convert(request.getOffice(), FormConverters.OFFICES);
                if (office.size() == 0) {
                    result = ValidationResult.merge(result, ValidationResult.error("office", "Unknown office"));
                } else if (office.size() > 1) {
                    result = ValidationResult.merge(result, ValidationResult.error("office", "Only one office per request"));
                } else {
                    long count = event.getResources().stream()
                            .filter(r -> r.getOfficeId() == office.get(0))
                            .count();
                    if (count == 0) {
                        result = ValidationResult.merge(result, ValidationResult.error("office", "No resources in the office"));
                    } else if (count > 1) {
                        result = ValidationResult.merge(result, ValidationResult.error("office", "More than one resource in the office"));
                    }
                }
            }
            if (organizer != null && !canModify(event, organizer)) {
                result = ValidationResult.merge(result, ValidationResult.error("calendarUrl", "No permissions to modify event"));
            }
            try {
                ttl = Integer.parseInt(request.getTtl());
            } catch (Exception e) {
                return ValidationResult.error("ttl", "Value should be numeric");
            }
        } catch (CalendarException ce) {
            result = ValidationResult.merge(result, ValidationResult.error("calendarUrl", ce));
        } catch (Exception e) {
            result = ValidationResult.merge(result, ValidationResult.error("calendarUrl", ValidationResult.INVALID_URL));
        }
        return result;
    }

    public CalendarAction prepare(AddResourceRequest request, String organizer) {
        this.organizer = organizer;
        result = validate(request);
        return buildAction(descriptor, FormConverters.toScan(request), ttl, event);
    }

    public boolean canModify(Event event, String login) {
        if (login == null) {
            return false; // anonymous users can't modify an event
        }
        try {
            Event e = calendarWeb.getEvent(event.getEventId(), staffClient.getByLogin(login).getUid());
            return e.getActions().isInvite();
        } catch (CalendarException ce) {
            return false;
        }
    }

    public CalendarAction buildAction(EventResourceDescriptor descriptor, Contexts.Scan context, int ttl, Event event) {
        CalendarAction action = new CalendarAction();
        action.setContext(context.toString());
        action.setEmail(descriptor.getLogin() + AbstractMessageBuilder.AT_YANDEX);
        action.setCreateDate(new Date());
        action.setStart(event.getStart());
        action.setName(event.getName());
        action.setSequence(event.getSequence());
        action.setInstanceStartTs(descriptor.getInstanceTs());
        action.setTriggerTime(minusMinutes(event.getStart(), ttl));
        action.setType(ActionType.ADD_RESOURCE);
        action.setStatus(Status.ACCEPTED);
        action.setEventId(descriptor.getEventId());
        action.setActionId(UUID.randomUUID().toString());
        return action;
    }

    public boolean isValid() {
        return result.getStatus() == ValidationResult.Status.OK;
    }
}
