package ru.yandex.qe.mail.meetings.ws;

import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

/**
 * @author Sergey Galyamichev
 */
@Component
public class TaskExecutor {
    private static final String MEETINGS_EXECUTOR_THREAD = "MeetingsExecutorThread #";
    private static final AtomicInteger counter = new AtomicInteger();
    private static final long TTL = TimeUnit.DAYS.toMillis(1);

    private static final String COMPLETED = "completed";
    private static final String RUNNING = "running";
    private static final String FAILED = "failed %s";
    private static final String UNTRACKED = "untracked";

    @Value("${executor.thread.count:3}")
    private int threadCount;
    private ScheduledExecutorService service = Executors.newScheduledThreadPool(threadCount, TaskExecutor::newThread);
    private Map<String, State> tasks = new ConcurrentHashMap<>();

    public TaskExecutor() {
        service.scheduleAtFixedRate(this::cleanUp, 0, 1, TimeUnit.MINUTES);
    }

    private void cleanUp() {
        tasks.entrySet().removeIf(e -> e.getValue().time + TTL < System.currentTimeMillis());
    }

    public String run(Runnable task) {
        String taskId = UUID.randomUUID().toString();
        service.submit(new TaskWrapper(task, taskId));
        return taskId;
    }

    public String status(String taskId) {
        State state = tasks.get(taskId);
        if (state != null) {
            return state.status;
        } else {
            return UNTRACKED;
        }
    }

    static class State {
        final long time;
        final String status;
        State(String status) {
            time = System.currentTimeMillis();
            this.status = status;
        }
    }

    class TaskWrapper implements Runnable {
        private final Runnable subTask;
        private final String taskId;

        TaskWrapper(Runnable subTask, String taskId) {
            this.subTask = subTask;
            this.taskId = taskId;
        }

        @Override
        public void run() {
            updateState(taskId, RUNNING);
            try {
                subTask.run();
            } catch (Exception e) {
                updateState(taskId, String.format(FAILED, e.getMessage()));
            }
            updateState(taskId, COMPLETED);
        }
    }

    private void updateState(String taskId, String newStatus) {
        tasks.compute(taskId, (k, old) -> new State(newStatus));
    }

    private static Thread newThread(Runnable r) {
        Thread t = new Thread(r);
        t.setDaemon(true);
        t.setName(MEETINGS_EXECUTOR_THREAD + counter.getAndIncrement());
        return t;
    }

}
