package ru.yandex.qe.mail.meetings.ws.controllers;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.ws.rs.ClientErrorException;
import javax.ws.rs.ServerErrorException;
import javax.ws.rs.core.Response;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.qe.mail.meetings.api.resource.MeetingsApiController;
import ru.yandex.qe.mail.meetings.api.resource.dto.ResourceSearchRequest;
import ru.yandex.qe.mail.meetings.api.resource.dto.ResourceSearchStatusResponse;
import ru.yandex.qe.mail.meetings.services.calendar.CalendarWeb;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Event;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Office;
import ru.yandex.qe.mail.meetings.services.calendar.dto.faults.CalendarException;
import ru.yandex.qe.mail.meetings.services.staff.StaffClient;
import ru.yandex.qe.mail.meetings.services.staff.dto.Person;
import ru.yandex.qe.mail.meetings.ws.services.FindResourceService;

/**
 * @author Stepan Kladikov
 */

@Service("meetingsApi")
public class MeetingsApiControllerImpl implements MeetingsApiController {
    private static final Logger LOG = LoggerFactory.getLogger(MeetingsApiController.class);

    LoadingCache<Integer, Office> officesCache = CacheBuilder.newBuilder()
            .expireAfterAccess(7, TimeUnit.DAYS)
            .weakValues()
            .build(new CacheLoader<>() {
                @Override
                public Office load(@Nonnull Integer key) {
                    LOG.debug("Load actual offices list from calendar");
                    List<Office> newOffices = calendarWeb.getOffices().getOffices();
                    Map<Integer, Office> newOfficesData = Maps.uniqueIndex(newOffices, Office::getId);
                    LOG.debug("Loaded offices: {}", newOffices.size());
                    officesCache.putAll(newOfficesData);

                    return newOfficesData.get(key);
                }
            });

    @Autowired
    private CalendarWeb calendarWeb;
    @Autowired
    private StaffClient staffClient;
    @Autowired
    private FindResourceService findResourceService;

    @Override
    public String findResources(ResourceSearchRequest data) {
        LOG.debug("findResource(eventId = {}, officeId = {}, organizer = {})", data.getEventId(), data.getOfficeIds(), data.getOrganizer());
        Person user = tryFetchPerson(data.getOrganizer());
        Event event = tryFetchEventById(data.getEventId(), user.getUid());

        if (!event.getActions().isInvite()) {
            throw new ClientErrorException(Response.Status.FORBIDDEN);
        }

        List<Office> offices = tryFetchOffices(data.getOfficeIds());

        return findResourceService.findResources(user, event, offices);
    }

    @Override
    public ResourceSearchStatusResponse resourceSearchStatus(Integer eventId) {
        return findResourceService.searchStatus(eventId);
    }

    private Person tryFetchPerson(String login) {
        Person user = staffClient.getByLogin(login);

        if (user == null) {
            throw new ClientErrorException("User does not found", Response.Status.NOT_FOUND);
        }

        return user;
    }

    private List<Office> tryFetchOffices(Set<Integer> officeIds) {
        try {
            return officeIds.stream()
                    .map(officeId -> officesCache.getUnchecked(officeId))
                    .collect(Collectors.toList());
        } catch (CacheLoader.InvalidCacheLoadException e) {
            throw new ClientErrorException(e.getMessage(), Response.Status.NOT_FOUND);
        }
    }

    private Event tryFetchEventById(int eventId, String uid) {
        try {
            return calendarWeb.getEvent(eventId, uid);
        } catch (CalendarException e) {
            if (e.getName().equals("event-not-found")) {
                throw new ClientErrorException(e.getMessage(), Response.Status.NOT_FOUND);
            } else {
                throw new ServerErrorException(e.getMessage(), Response.Status.INTERNAL_SERVER_ERROR);
            }
        } catch (Exception e) {
            throw new ServerErrorException(e.getMessage(), Response.Status.INTERNAL_SERVER_ERROR);
        }
    }
}
