package ru.yandex.qe.mail.meetings.ws.handlers;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.inject.Inject;

import com.codahale.metrics.MetricRegistry;
import org.joda.time.Duration;
import org.joda.time.Interval;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.stereotype.Service;

import ru.yandex.qe.mail.meetings.booking.BasicMeetingInfo;
import ru.yandex.qe.mail.meetings.booking.BookingResultMessageBuilder;
import ru.yandex.qe.mail.meetings.booking.FuzzyBookingService;
import ru.yandex.qe.mail.meetings.booking.TimeTable;
import ru.yandex.qe.mail.meetings.booking.dto.BookingResult;
import ru.yandex.qe.mail.meetings.booking.dto.FullSearchResult;
import ru.yandex.qe.mail.meetings.booking.impl.TimeTableImpl;
import ru.yandex.qe.mail.meetings.booking.util.BookingUtils;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Repetition;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Resource;
import ru.yandex.qe.mail.meetings.utils.RepetitionUtils;

@Service("bookingService")
public class BookingHandler extends FormHandler<FullSearchResult.MissingReason> {
    private static final Logger LOG = LoggerFactory.getLogger(BookingHandler.class);

    private static final ThreadLocal<SimpleDateFormat> SDF = ThreadLocal.withInitial(() -> new SimpleDateFormat("yyyy-MM-dd"));

    private static final Map<String, Integer> DAYS = new HashMap<>(){{
        put("Понедельник", Calendar.MONDAY);
        put("Вторник", Calendar.TUESDAY);
        put("Среда", Calendar.WEDNESDAY);
        put("Четверг", Calendar.THURSDAY);
        put("Пятница", Calendar.FRIDAY);
    }};

    @Nonnull
    private final FuzzyBookingService fuzzyBookingService;

    @Nonnull
    private final JavaMailSender mailSender;

    @Nonnull
    private final BookingResultMessageBuilder messageBuilder;

    @Inject
    public BookingHandler(
            @Nonnull FuzzyBookingService fuzzyBookingService,
            @Nonnull JavaMailSender mailSender,
            @Nonnull BookingResultMessageBuilder messageBuilder,
            @Nonnull MetricRegistry metricRegistry) {
        super(metricRegistry);
        this.fuzzyBookingService = fuzzyBookingService;
        this.mailSender = mailSender;
        this.messageBuilder = messageBuilder;
    }

    @Override
    protected BookingResult doAction(@Nonnull String organizer, @Nonnull Map<String, String> bookingRequest) throws Exception {
        var builder = new BookingRequestBuilder();
        bookingRequest.forEach(builder::apply);

        var request = builder.build(organizer);
        LOG.info("[MARK][REQUEST_VALIDATED] request was build");
        // отправим данние из заполса в голован для дальнейшей аналитики по кейсам
        dumpRequestInfo(request);

        return fuzzyBookingService.bookMeeting(
                request.timeRestrictions,
                request.duration,
                request.meetingInfo,
                request.attendees,
                request.additionalAttendees,
                request.roomRestrictions,
                request.allowMeetingsWithoutRooms,
                request.repetition,
                request.notifyBefore
        );
    }

    @Override
    void handleError(@Nonnull String organizer, @Nonnull Map<String, String> bookingRequest, @Nullable HandlerResult<FullSearchResult.MissingReason> result) {
        var reason = Optional.ofNullable(result).map(r -> r.error()).orElse(FullSearchResult.MissingReason.INTERNAL_ERROR);
        mailSender.send(mm -> messageBuilder.prepareMessage(mm, organizer, bookingRequest, reason, false));
        if (reason == FullSearchResult.MissingReason.INTERNAL_ERROR) {
            mailSender.send(mm -> messageBuilder.prepareMessage(mm, "calendar-assistant-errors", bookingRequest, reason, true   ));
        }
    }

    @Override
    void handleSuccess(@Nonnull String organizer, @Nonnull Map<String, String> request, @Nullable HandlerResult<FullSearchResult.MissingReason> result) {

    }

    private void dumpRequestInfo(Request request) {
        registry().counter(MetricRegistry.name(BookingHandler.class, request.repetition == null ? "single" : "regular")).inc();
        if (!request.allowMeetingsWithoutRooms) {
            registry().counter(MetricRegistry.name(BookingHandler.class, "room", "required")).inc();
        }
        registry().histogram(MetricRegistry.name(BookingHandler.class, "meeting", "attendees")).update(request.attendees.size());
        registry().histogram(MetricRegistry.name(BookingHandler.class, "meeting", "duration")).update(request.duration.toStandardMinutes().getMinutes());
    }

    private static class BookingRequestBuilder extends RequestBuilder<Request> {

        @Override
        protected Request build(@Nonnull String organizer) throws Exception {
            var isRegular = checkRequired("meeting_type", "регулярной");

            var timeRestrictions = buildTimeTable(isRegular);
            var duration = buildDuration();
            var attendees = buildAttendees("participants");
            var additionalAttendees = buildAttendees("additional_attendees");
            var roomRestrictions = buildRoomRestrictions();
            var allowMeetingsWithoutRooms = isRegular || buildAllowMeetingsWithoutRooms();
            var repetition = isRegular ? buildRepetition() : null;

            return new Request(
                    organizer,
                    timeRestrictions,
                    duration,
                    attendees,
                    additionalAttendees,
                    roomRestrictions,
                    allowMeetingsWithoutRooms,
                    repetition,
                    getOrDefault("meeting_name", "Без названия"),
                    getOrDefault("meeting_desc", ""),
                    checkOptional("participantsCanInvite", "Нет", "Да"),
                    checkOptional("participantsCanEdit", "Нет", "Да"),
                    !checkOptional("othersCanNotView", "Нет", "Да"),
                    Integer.parseInt(getOrDefault("ttl_booking", "0")));
        }
        private Repetition buildRepetition() {
            var period = getRequired("period");
            switch (period) {
                case "Раз в неделю":
                    return RepetitionUtils.buildRepetition(Repetition.Type.WEEKLY, 1);
                case "Раз в 2 недели":
                    return RepetitionUtils.buildRepetition(Repetition.Type.WEEKLY, 2);
                case "Раз в месяц":
                    return RepetitionUtils.buildRepetition(Repetition.Type.MONTHLY_NUMBER, 1);
                default:
                    throw new IllegalArgumentException("unexpected period: " + period);

            }
        }

        private boolean buildAllowMeetingsWithoutRooms() {
            return checkOptional("book_now",  "Нет", "Нет");
        }

        private Resource.Info buildRoomRestrictions() {
            var rr = new Resource.Info();

            Arrays.stream(
                    getOrDefault("room-params", "").split(",")
            )
                    .map(String::trim)
                    .filter(s -> !s.isEmpty())
                    .map(String::toLowerCase)
                    .forEach(s -> {
                        switch (s) {
                            case "конференц-связь":
                                LOG.debug("conf is required");
                                rr.setVoiceConferencing(true);
                                break;
                            case "видеосвязь":
                                LOG.debug("video is required");
                                rr.setHasVideo(true);
                                break;
                            case "жк-панель":
                                LOG.debug("lcd-panel is required");
                                rr.setLcdPanel(1);
                                break;
                            default:
                                LOG.info("ignoring param {}", s);
                                break;
                        }
                    });
            return rr;
        }

        private Set<String> buildAttendees(@Nonnull String fieldName) {
            //  строка вида: 'Никифоров Игорь (inikifor), Кокарев Вадим (vkokarev)'
            var raw = getOrDefault(fieldName, "");
            return Arrays.stream(raw.split(","))
                    .filter(s -> !s.isEmpty())
                    .map(s -> {
                        // Тут могут быть девичьи фамилии, логин всегда идет в конце
                        var parts = s.split("\\(");
                        return parts[parts.length - 1];
                    })
                    .map(s -> s.substring(0, s.length() - 1))
                    .collect(Collectors.toSet());
        }

        private Duration buildDuration() {
            final String targetField = checkRequired("duration", "другая") ? "own_duration_minutes" : "duration";
            return Duration.standardMinutes(Long.parseLong(getRequired(targetField)));
        }

        private TimeTable buildTimeTable(boolean isRegular) throws ParseException {
            return isRegular ? buildRegularTimeTable() : buildSingleTimeTable();
        }

        private TimeTable buildRegularTimeTable() throws ParseException {
            // строим временные ограничения для регулярных встреч

            // получаем дни, в которые может пройти встреча
            var isAnyDay = checkRequired("day_type", "любой день");
            Collection<Integer> availableDays = isAnyDay ? DAYS.values() : parseDays();

            // теперь нужно наложить сверху ограничения по дням
            var l = hhssToMs(getRequired("early_border"));
            var r = hhssToMs(getRequired("late_border"));
            assert  l <= r;

            // выставляем левую границу для встречи
            // это либо текущий момент, либо значение опции regular_early_border
            var earlyDayBorder = Optional.ofNullable(getOrDefault("regular_early_border", null))
                    .map(dayString -> {
                        try {
                            var df = SDF.get();
                            return df.parse(dayString);
                        } catch (ParseException e) {
                            throw new RuntimeException(e);
                        }
                    }).orElse(new Date(System.currentTimeMillis()));

            // для регулярной встречи нужно построить расписание на следуюущие 7 дней, далее оно будет эктраполироваться
            var dailyRestrictions = IntStream.range(1,8)
                    .mapToObj(shift -> {
                        // получаем начало дня, который будет через shift дней
                        var cal = Calendar.getInstance();
                        // используем левую границу для размещения встреч, как левую границу для временных ограничений
                        cal.setTime(earlyDayBorder);
                        cal.set(Calendar.HOUR_OF_DAY, 0);
                        cal.set(Calendar.MINUTE, 0);
                        cal.set(Calendar.SECOND, 0);
                        cal.set(Calendar.MILLISECOND, 0);
                        cal.add(Calendar.DAY_OF_YEAR, shift);
                        return cal;
                    })
                    // для каждого дня создаем интервалы занятости
                    .flatMap(cal -> {
                        var result = new ArrayList<Interval>(3);
                        var ts = cal.getTimeInMillis();
                        if (availableDays.contains(cal.get(Calendar.DAY_OF_WEEK))) {
                            // день среди доступных, просто добавим ограничения "не раньше" и "не позже"
                            // это от 00:00 до leftBorder
                            result.add(new Interval(ts, ts + l));
                            // это от right border до 23:59
                            result.add(new Interval(ts + r, ts + Duration.standardDays(1).getMillis()));
                        } else {
                            // день не доступен для встреч, нужно его полностью исключить
                            result.add(new Interval(ts, ts + Duration.standardDays(1).getMillis()));
                        }
                        return result.stream();
                    })
                    .map(BookingUtils::intervalable)
                    .collect(Collectors.toUnmodifiableList());

            var terms = BookingUtils.termsFromData(dailyRestrictions);
            return TimeTableImpl.fromEventDates(terms, dailyRestrictions);
        }

        private Set<Integer> parseDays() {
            return Arrays.stream(
                    getOrDefault("requested_days", "").split(",")
            )
                    .map(String::trim)
                    .filter(s -> !s.isEmpty())
                    .map(DAYS::get)
                    .collect(Collectors.toUnmodifiableSet());
        }

        private TimeTable buildSingleTimeTable() throws ParseException {
            // первый этап - получить общие ограничения по датам

            String dates;
            if (!"".equals(state().getOrDefault("dates_interval", ""))) {
                // dates - строка вида '2019-12-18 - 2019-12-19'
                dates = getRequired("dates_interval");
            } else {
                // а тут конкретная дата
                var d = getRequired("dates_exact");
                dates = d + " - " + d;
            }

            var parts = dates.split(" - ");
            assert parts.length == 2;

            var df = SDF.get();
            // todo написать нормально
            var start = df.parse(parts[0]);

            // накидываем к концу интревала сутки, т.к. хотим инклюзивный интервал
            var end = new Date(df.parse(parts[1]).getTime() + Duration.standardDays(1).getMillis());
            if (end.getTime() < System.currentTimeMillis()) {
                throw new IllegalArgumentException("can't create meeting in past");
            }

            assert start.getTime() < end.getTime();

            var terms = new Interval(
                    start.getTime() > System.currentTimeMillis() ? start.getTime() : roundUp(System.currentTimeMillis()),
                    end.getTime()
            );
            // todo сдеать terms строго в будущем

            // теперь нужно наложить сверху ограничения по дням
            var l = hhssToMs(getRequired("early_border"));
            var r = hhssToMs(getRequired("late_border"));
            assert  l <= r;

            // range переьбирает смещение в днях от start до end
            var dailyRestrictions = IntStream.range(
                    0,
                    (int)TimeUnit.DAYS.convert(
                            end.getTime() - start.getTime(),
                            TimeUnit.MILLISECONDS) + 2
            )
                    .mapToObj(shift -> start.getTime() + shift * Duration.standardDays(1).getMillis())
                    // для каждого дня создаем интервалы занятости
                    .flatMap(ts -> {
                        var result = new ArrayList<Interval>(3);

                        // это от 00:00 до leftBorder
                        result.add(new Interval(ts, ts + l));

                        // это от right border до 23:59
                        result.add(new Interval(ts + r, ts + Duration.standardDays(1).getMillis()));

                        // теперь добавим целый день, если это выходной
                        var cal = Calendar.getInstance();
                        cal.setTimeInMillis(ts);
                        var dayOfWeek = cal.get(Calendar.DAY_OF_WEEK);

                        if (dayOfWeek == Calendar.SATURDAY || dayOfWeek == Calendar.SUNDAY) {
                            result.add(new Interval(ts, ts + Duration.standardDays(1).getMillis()));
                        }

                        return result.stream();
                    })
                    .map(BookingUtils::intervalable)
                    .collect(Collectors.toUnmodifiableList());


            return TimeTableImpl.fromEventDates(
                    terms,
                    dailyRestrictions
            );

        }

        private long roundUp(long t) {
            var calendar = Calendar.getInstance();
            calendar.setTimeInMillis(t);
            calendar.set(Calendar.MILLISECOND, 0);
            calendar.set(Calendar.SECOND, 0);
            calendar.set(Calendar.MINUTE, 0);
            calendar.add(Calendar.HOUR, 1);
            return calendar.getTimeInMillis();
        }

        private long hhssToMs(@Nonnull String s) {
            var parts = s.split(":");
            assert parts.length == 2;
            var hours = Long.parseLong(parts[0]);
            var minutes = Long.parseLong(parts[1]);
            return TimeUnit.MILLISECONDS.convert(hours * 60 + minutes, TimeUnit.MINUTES);
        }
    }

    private static class Request {
        private final TimeTable timeRestrictions;
        private final Duration duration;
        private final Set<String> attendees;
        private final Set<String> additionalAttendees;
        private final Resource.Info roomRestrictions;
        private final boolean allowMeetingsWithoutRooms;
        private final Repetition repetition;
        private final int notifyBefore;
        public BasicMeetingInfo meetingInfo;


        private Request(
                String organizer,
                TimeTable timeRestrictions,
                Duration duration,
                Set<String> attendees,
                Set<String> additionalAttendees,
                Resource.Info roomRestrictions,
                boolean allowMeetingsWithoutRooms,
                Repetition repetition,
                String name,
                String desc,
                boolean participantsCanInvite,
                boolean participantsCanEdit,
                boolean othersCanView,
                int notifyBefore) {
            this.timeRestrictions = timeRestrictions;
            this.duration = duration;
            this.attendees = attendees;
            this.additionalAttendees = additionalAttendees;
            this.roomRestrictions = roomRestrictions;
            this.allowMeetingsWithoutRooms = allowMeetingsWithoutRooms;
            this.repetition = repetition;
            this.notifyBefore = notifyBefore;
            this.meetingInfo = new BasicMeetingInfo(
                    organizer,
                    name,
                    desc,
                    participantsCanInvite,
                    participantsCanEdit,
                    othersCanView
            );
        }
    }
}
