package ru.yandex.qe.mail.meetings.ws.handlers;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.codahale.metrics.MetricRegistry;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class FormHandler<ReasonType> {
    private static final Logger LOG = LoggerFactory.getLogger(FormHandler.class);

    private static final ObjectMapper MAPPER = new ObjectMapper();

    @Nonnull
    private final MetricRegistry metricRegistry;


    public void process(@Nonnull String organizer, Map<String, String> request) throws JsonProcessingException {
        final var klz = this.getClass();
        LOG.info("[MARK][START] start {} request from {}", klz.getSimpleName(), organizer);
        metricRegistry.counter(MetricRegistry.name(klz, "start")).inc();

        var info = MAPPER.writeValueAsString(
                request
                        .entrySet()
                        .stream()
                        .filter(e -> e.getValue() != null && !"null".equals(e.getValue()))
                        .collect(Collectors.toMap(
                                Map.Entry::getKey,
                                Map.Entry::getValue
                        ))
        );
        LOG.info("[MARK][INPUT][{}]", info);

        HandlerResult<ReasonType> result = null;
        try {
            result = doAction(organizer, request);
        } catch (Exception e) {
            LOG.error("[MARK][CRASH] " + klz.getSimpleName() + " crashed", e);
            LOG.error("RESULT: {} for [organizer = {}, request = {}]", e.getMessage(), organizer, request);

            metricRegistry.counter(MetricRegistry.name(klz, "crash")).inc();
            metricRegistry.counter(MetricRegistry.name(klz, "crash", e.getClass().getSimpleName())).inc();
        }

        if (Optional.ofNullable(result).map(HandlerResult::isOk).orElse(false)) {
            LOG.info("[MARK][SUCCESS]");
            registry().counter(MetricRegistry.name(klz, "success")).inc();
            handleSuccess(organizer, request, result);
        } else {
            var reason = Optional.ofNullable(result).map(HandlerResult::metricReason).orElse("INTERNAL_ERROR");
            LOG.info("[MARK][FAILED][{}] {} for {} failed", reason, klz.getSimpleName(), organizer);
            registry().counter(MetricRegistry.name(klz, "fail")).inc();
            registry().counter(MetricRegistry.name(klz, "fail", reason)).inc();
            handleError(organizer, request, result);
        }
    }

    abstract protected HandlerResult<ReasonType> doAction(@Nonnull String organizer, @Nonnull Map<String, String> request) throws Exception;

    abstract void handleError(@Nonnull String organizer, @Nonnull Map<String, String> request, @Nullable HandlerResult<ReasonType> result);

    abstract void handleSuccess(@Nonnull String organizer, @Nonnull Map<String, String> request, @Nullable HandlerResult<ReasonType> result);

    protected FormHandler(@Nonnull MetricRegistry metricRegistry) {
        this.metricRegistry = metricRegistry;
    }

    protected MetricRegistry registry() {
        return metricRegistry;
    }

    protected abstract static class RequestBuilder<RequestType> {
        private final Map<String, String> state = new HashMap<>();

        protected abstract RequestType build(@Nonnull String organizer) throws Exception;

        protected String getOrDefault(@Nonnull String varName, @Nullable String defaultValue) {
            return Optional.ofNullable(state.get(varName.toLowerCase())).orElse(defaultValue);
        }

        protected boolean checkRequired(@Nonnull String varName, @Nonnull String expectedValue) {
            return expectedValue.equalsIgnoreCase(getRequired(varName));
        }

        protected boolean checkOptional(@Nonnull String varName, @Nonnull String defaultValue, @Nonnull String expectedValue) {
            return expectedValue.equalsIgnoreCase(getOrDefault(varName, defaultValue));
        }

        protected String getRequired(@Nonnull String varName) {
            return Optional.ofNullable(state.get(varName.toLowerCase())).orElseThrow(() -> new IllegalStateException("required field " + varName + " is missing"));
        }

        protected Map<String, String> state() {
            return state;
        }

        protected void apply(String slug, String value) {
            if (state.put(slug.toLowerCase(), value) != null) {
                throw new RuntimeException("unexpected prop (" + slug + ") override");
            }
        }
    }
}
