package ru.yandex.qe.mail.meetings.ws.handlers;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.inject.Inject;

import com.codahale.metrics.MetricRegistry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.stereotype.Service;

import ru.yandex.qe.mail.meetings.services.calendar.CalendarWeb;
import ru.yandex.qe.mail.meetings.services.staff.StaffClient;
import ru.yandex.qe.mail.meetings.synchronizer.MeetingSynchronizer;
import ru.yandex.qe.mail.meetings.synchronizer.SyncResultMessageBuilder;
import ru.yandex.qe.mail.meetings.synchronizer.dto.SyncErrors;
import ru.yandex.qe.mail.meetings.synchronizer.impl.AbcServices;
import ru.yandex.qe.mail.meetings.synchronizer.impl.StaffGroups;
import ru.yandex.qe.mail.meetings.ws.EventResourceDescriptor;


@Service("meetingSynchronizerHandler")
public class MeetingSynchronizerHandler extends FormHandler<SyncErrors> {
    private static final Logger LOG = LoggerFactory.getLogger(MeetingSynchronizerHandler.class);

    @Nonnull
    private final MeetingSynchronizer meetingSynchronizer;

    @Nonnull
    private final AbcServices abcServices;

    @Nonnull
    private final StaffGroups staffGroups;

    @Nonnull
    private final StaffClient staffClient;

    @Nonnull
    private final CalendarWeb calendarWeb;

    @Nonnull
    private final SyncResultMessageBuilder messageBuilder;

    @Nonnull
    private final JavaMailSender mailSender;

    @Inject
    public MeetingSynchronizerHandler(@Nonnull MeetingSynchronizer meetingSynchronizer, @Nonnull MetricRegistry metricRegistry, @Nonnull AbcServices abcServices, @Nonnull StaffGroups staffGroups, @Nonnull StaffClient staffClient, @Nonnull CalendarWeb calendarWeb, @Nonnull SyncResultMessageBuilder messageBuilder, @Nonnull JavaMailSender mailSender) {
        super(metricRegistry);
        this.meetingSynchronizer = meetingSynchronizer;
        this.abcServices = abcServices;
        this.staffGroups = staffGroups;
        this.staffClient = staffClient;
        this.calendarWeb = calendarWeb;
        this.messageBuilder = messageBuilder;
        this.mailSender = mailSender;
    }

    @Override
    protected HandlerResult<SyncErrors> doAction(@Nonnull String organizer, @Nonnull Map<String, String> syncRequest) throws Exception {
        var builder = new SyncRequestBuilder();
        syncRequest.forEach(builder::apply);

        var request = builder.build(organizer);
        LOG.info("[MARK][REQUEST_VALIDATED] request was build");

        var abcServiceList = request.abcServices.stream().map(s -> abcServices.byName(s, "ru").get()).collect(Collectors.toList());
        var staffGroupList = request.staffGroups.stream().map(g -> staffGroups.byName(g, "ru").get()).collect(Collectors.toList());
        var personList = request.personLogins.stream().map(staffClient::getByLogin).collect(Collectors.toList());

        EventResourceDescriptor descriptor = EventResourceDescriptor.fromRequest(calendarWeb, staffClient, organizer, request.eventUrl);
        LOG.info("{} -> {}", request.eventUrl, descriptor.getEventId());

        return meetingSynchronizer.assign(descriptor.getEventId(), request.organizer, abcServiceList, staffGroupList, personList);
    }

    @Override
    void handleError(@Nonnull String organizer, @Nonnull Map<String, String> request, @Nullable HandlerResult<SyncErrors> result) {
        var reason = Optional.ofNullable(result).map(r -> r.error()).orElse(SyncErrors.UNKNOWN);
        mailSender.send(mm -> messageBuilder.prepareErrorMessage(mm, organizer, request, reason, false));
        if (reason == SyncErrors.UNKNOWN) {
            mailSender.send(mm -> messageBuilder.prepareErrorMessage(mm, "calendar-assistant-errors", request, reason, true));
        }
    }

    @Override
    void handleSuccess(@Nonnull String organizer, @Nonnull Map<String, String> request, @Nullable HandlerResult<SyncErrors> result) {
        mailSender.send(mm -> messageBuilder.prepareOkMessage(mm, organizer, request));
    }

    private static class SyncRequest {
        private final String organizer;
        private final List<String> abcServices;
        private final List<String> staffGroups;
        private final List<String> personLogins;
        private final String eventUrl;

        private SyncRequest(String organizer, String eventUrl, List<String> abcServices, List<String> staffGroups, List<String> personLogins) {
            this.organizer = organizer;
            this.abcServices = abcServices;
            this.eventUrl = eventUrl;
            this.staffGroups = staffGroups;
            this.personLogins = personLogins;
        }
    }

    private static class SyncRequestBuilder extends RequestBuilder<SyncRequest> {

        @Override
        protected SyncRequest build(@Nonnull String organizer) {
            var abcServices = splitAndGet("assign_abc_services");
            var staffGroups = splitAndGet("assign_staff_groups");
            var personLogins = splitAndGet("assign_person_logins")
                    .stream()
                    .map(s -> {
                        // Тут могут быть девичьи фамилии, логин всегда идет в конце
                        var parts = s.split("\\(");
                        return parts[parts.length - 1];
                    })
                    .map(s -> s.substring(0, s.length() - 1))
                    .sorted()
                    .distinct()
                    .collect(Collectors.toList());

            var eventUrl = getRequired("calendar_url_sync");
            return new SyncRequest(organizer, eventUrl, abcServices, staffGroups, personLogins);
        }

        private List<String> splitAndGet(String field) {
            //e.g.: Умная реклама, Мобильная Морда
            return Optional.ofNullable(getOrDefault(field, null))
                    .map(s -> s.split(","))
                    .map(Arrays::asList)
                    .map(l -> l.stream().map(String::strip).collect(Collectors.toList()))
                    .orElseGet(Collections::emptyList);
        }
    }
}
