package ru.yandex.qe.mail.meetings.ws.mock;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Sergey Galyamichev
 */
public class JsonDiff {
    private static final Logger LOG = LoggerFactory.getLogger(JsonDiff.class);
    private static final Map<Integer, Map<String, String>> EVENTS = new HashMap<>();
    private static final ObjectMapper OM = new ObjectMapper();

    private static Map<String, String> dump(JsonNode nodes, String prefix, Map<String, String> result) {
        Iterator<Map.Entry<String, JsonNode>> it = nodes.fields();
        while (it.hasNext()) {
            Map.Entry<String, JsonNode> field = it.next();
            if (field.getValue().isValueNode()) {
                result.put(prefix + "." + field.getKey(), field.getValue().asText());
            } else if (field.getValue().isObject()) {
                dump(field.getValue(), prefix + "." + field.getKey(), result);
            } else if (field.getValue().isArray()) {
                Iterator<JsonNode> elements = field.getValue().elements();
                while (elements.hasNext()) {
                    JsonNode element = elements.next();
                    int hash = element.toString().hashCode();
                    dump(element, prefix + "." + field.getKey() + hash, result);
                }
            } else if (field.getValue().isMissingNode()) {
                result.put(prefix + "." + field.getKey(), field.getValue().textValue());
            }
        }
        return result;
    }

    public static void add(String message) {
        try {
            if (message.startsWith("<result>")) {
                JsonNode node = OM.readTree(message.substring("<result>".length(), message.length() - "</result>".length()));
                addNode(node);
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public static JsonNode addNode(JsonNode node) {
        int id = node.path("id").intValue();
        EVENTS.put(id, dump(node, "event", new HashMap<>()));
        return node;
    }

    public static void check(String message) {
        try {
            JsonNode node = OM.readTree(message);
            checkNode(node);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public static void checkNode(JsonNode node) {
        Map<String, String> result = dump(node, "event", new HashMap<>());
        int id = node.path("id").intValue();
        if (EVENTS.containsKey(id)) {
            Map<String,String> old = new HashMap<>(EVENTS.get(id));
            old.keySet().removeAll(result.keySet());
            checkRemoved(parseDiff(old));
            result.keySet().removeAll(EVENTS.get(id).keySet());
            checkAdded(parseDiff(result));
            EVENTS.remove(id);
        } else {
            LOG.warn("Can't check before add");
        }
    }

    private static void checkAdded(Map<String, Map<String, String>> added) {
        if (!added.isEmpty()) {
            throw new IllegalStateException("New fields are appeared: " + added);
        }
    }

    private static void checkRemoved(Map<String, Map<String, String>> removed) {
        List<String> participants = new ArrayList<>();
        for (Map.Entry<String, Map<String, String>> entry : removed.entrySet()) {
            if (entry.getKey().startsWith("event.attendees")) {
                participants.add(entry.getValue().get("login"));
            } else {
                throw new IllegalStateException("Removed unexpected field: " + entry);
            }
        }
        LOG.info("Removed participant: " + participants);
    }

    private static Map<String, Map<String, String>> parseDiff(Map<String, String> diff) {
        Map<String, Map<String, String>> result = new HashMap<>();
        for (Map.Entry<String, String> entry : diff.entrySet()) {
            String record = entry.getKey();
            String object = record.substring(0, record.lastIndexOf("."));
            String field = record.substring(record.lastIndexOf(".") + 1);
            result.computeIfAbsent(object, k -> new HashMap<>()).put(field, entry.getValue());
        }
        return result;
    }
}
