package ru.yandex.qe.mail.meetings.ws.services;

import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import javax.ws.rs.ClientErrorException;
import javax.ws.rs.core.Response;

import com.google.common.base.Joiner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.qe.mail.meetings.api.resource.dto.ActionType;
import ru.yandex.qe.mail.meetings.api.resource.dto.CalendarAction;
import ru.yandex.qe.mail.meetings.api.resource.dto.ResourceSearchStatusResponse;
import ru.yandex.qe.mail.meetings.api.resource.dto.Status;
import ru.yandex.qe.mail.meetings.cron.actions.Contexts;
import ru.yandex.qe.mail.meetings.rooms.dao.SentEmailsDao;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Event;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Office;
import ru.yandex.qe.mail.meetings.services.staff.dto.Person;

import static ru.yandex.qe.mail.meetings.cron.AbstractMessageBuilder.AT_YANDEX;
import static ru.yandex.qe.mail.meetings.utils.DateUtils.minusMinutes;

@Service
public class FindResourceService {
    private static final Logger LOG = LoggerFactory.getLogger(FindResourceService.class);
    private static final int DEFAULT_TTL_MINUTES = 15;

    @Autowired
    private SentEmailsDao sentEmails;

    public String findResources(Person user, Event event, List<Office> offices) {
        CalendarAction action = buildAction(user, event, offices);

        List<CalendarAction> runningActions = sentEmails.getActions(action.getEventId(), ActionType.ADD_RESOURCE)
                .stream().filter(a -> a.getStatus() == Status.ACCEPTED).collect(Collectors.toList());

        runningActions.forEach(a -> sentEmails.updateAction(a.getActionId(), Status.DECLINED));

        if (!offices.isEmpty()) {
            sentEmails.insertActions(Collections.singleton(action));
        }

        return "OK";
    }

    public ResourceSearchStatusResponse searchStatus(Integer eventId) {
        List<CalendarAction> existingActions = sentEmails.getActions(eventId, ActionType.ADD_RESOURCE);

        if (existingActions.isEmpty()) {
            throw new ClientErrorException("Searches for this event did not started", Response.Status.NOT_FOUND);
        } else {
            existingActions.sort(Comparator.comparingInt(CalendarAction::getId));
            CalendarAction last = existingActions.get(existingActions.size() - 1);
            String status = last.getStatus().name();
            Set<Integer> officeIds = Contexts.Scan.parse(last.getContext()).getOfficeIds();

            return new ResourceSearchStatusResponse(status, officeIds);
        }
    }

    private CalendarAction buildAction(Person user, Event event, List<Office> offices) {
        List<Integer> officeIds = offices.stream().map(Office::getId).collect(Collectors.toList());
        String scanContext = Joiner.on(",").join(officeIds);
        Date triggerTime = minusMinutes(event.getStart(), DEFAULT_TTL_MINUTES);
        String email = user.getLogin() + AT_YANDEX;

        CalendarAction action = new CalendarAction();
        action.setContext(scanContext);
        action.setEmail(email);
        action.setCreateDate(new Date());
        action.setStart(event.getStart());
        action.setName(event.getName());
        action.setSequence(event.getSequence());
        action.setInstanceStartTs(event.getInstanceStartTs());
        action.setTriggerTime(triggerTime);
        action.setType(ActionType.ADD_RESOURCE);
        action.setStatus(Status.ACCEPTED);
        action.setEventId(event.getEventId());
        action.setActionId(UUID.randomUUID().toString());

        return action;
    }
}
