package ru.yandex.qe.mail.meetings.ws.sync;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.inject.Inject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.qe.mail.meetings.services.abc.dto.AbcService;
import ru.yandex.qe.mail.meetings.services.calendar.CalendarWeb;
import ru.yandex.qe.mail.meetings.services.staff.StaffClient;
import ru.yandex.qe.mail.meetings.services.staff.dto.Language;
import ru.yandex.qe.mail.meetings.services.staff.dto.Person;
import ru.yandex.qe.mail.meetings.services.staff.dto.StaffGroup;
import ru.yandex.qe.mail.meetings.synchronizer.dto.IdWithType;
import ru.yandex.qe.mail.meetings.synchronizer.dto.SyncEvent;
import ru.yandex.qe.mail.meetings.synchronizer.impl.AbcServices;
import ru.yandex.qe.mail.meetings.synchronizer.impl.StaffGroups;


@Service("syncHtmlBuilder")
public class SyncHtmlBuilder {
    private static final Logger LOG = LoggerFactory.getLogger(SyncHtmlBuilder.class);

    @Nonnull
    private final CalendarWeb calendarWeb;

    @Nonnull
    private final AbcServices abcServices;

    @Nonnull
    private final StaffClient staffClient;

    @Nonnull
    private final StaffGroups staffGroups;

    @Inject
    public SyncHtmlBuilder(@Nonnull CalendarWeb calendarWeb, @Nonnull AbcServices abcServices, @Nonnull StaffClient staffClient, @Nonnull StaffGroups staffGroups) {
        this.calendarWeb = calendarWeb;
        this.abcServices = abcServices;
        this.staffClient = staffClient;
        this.staffGroups = staffGroups;
    }

    public String toPretty(Map<SyncEvent, List<IdWithType>> eventToEntity) {
        if (eventToEntity.isEmpty()) {
            return "<div>Ничего не нашлось</div>";
        }
        return "<table>" +
                "<tr><td>Встреча</td><td>Участники</td><td>Автор</td><td>Отменить</td></tr>" +
                eventToEntity.entrySet().stream().map(e -> toPretty(e.getKey(), e.getValue())).collect(Collectors.joining()) +
                "</table>";
    }

    private String toPretty(SyncEvent syncEvent, List<IdWithType> entityList) {
        return "<tr>" +
                "<td>" + toPrettyEvent(syncEvent) + "</td>" +
                "<td>" + toPetty(syncEvent, entityList, true, ",<br>") + "</td>" +
                "<td>" + syncEvent.owner() + "@" + "</td>" +
                "<td>" + cancelLink(syncEvent, "", "Cancel") + "</td>" +
                "</tr>";
    }


    public String toPetty(SyncEvent syncEvent, Collection<? extends IdWithType> entityList, boolean addRmLink, String delimeter) {
        return entityList.stream().map(idWithType -> {
            String result;
            if (idWithType == IdWithType.NONE) {
                return "";
            }
            switch (idWithType.type) {
                case ABC:
                    var abcId = Integer.parseInt(idWithType.id);
                    result = abcServices.byId(abcId).map(this::abcLink).orElse("unknown");
                    break;
                case STAFF:
                    var staffId = Integer.parseInt(idWithType.id);
                    result = staffGroups.byId(staffId).map(this::groupLink).orElse("unknown");
                    break;
                case PERSON:
                    var person = staffClient.getByLogin(idWithType.id);
                    result = staffLink(person);
                    break;
                default:
                    result = "missing type " + idWithType;
                    break;
            }
            return result + (addRmLink ? " (" + rmLink(syncEvent, idWithType, "", "Удалить") + ")" : "");
        }).collect(Collectors.joining(delimeter));
    }

    public static String rmLink(SyncEvent syncEvent, IdWithType idWithType, String host, String label) {
        return "<a href=\"" + host + "/api/sync/unsubscribe?eventId=" + syncEvent.id() + "&groupId=" + idWithType.id + "&type=" + idWithType.type + "\">" + label + "</a>";
    }

    public static String cancelLink(SyncEvent syncEvent, String host, String label) {
        return "<a href=\"" + host + "/api/sync/cancel?eventId=" + syncEvent.id() + "\">" + label + "</a>";
    }

    private String abcLink(AbcService abcService) {
        return "<a href=\"https://abc.yandex-team.ru/services/" + abcService.slug() +"\">" + abcService.name("ru") + "</a>";
    }

    private String groupLink(StaffGroup staffGroup) {
        return "<a href=\"https://staff.yandex-team.ru/departments/" + staffGroup.url() +"\">" + staffGroup.name() + "</a>";
    }

    private String staffLink(Person person) {
        return "<a href=\"https://staff.yandex-team.ru/" + person.getLogin() +"\">" + person.getName().toString(Language.RUSSIAN) + " (" + person.getLogin() + "@)</a>";
    }

    private String toPrettyEvent(SyncEvent syncEvent) {
        try {
            var event = calendarWeb.getEvent(syncEvent.id());
            return "<a href=\"https://calendar.yandex-team.ru/event/" + syncEvent.id() +"\">" + event.getName() + "</a>";
        } catch (Exception e) {
            LOG.warn("didn't get event " + syncEvent.id(), e);
        }
        return "deleted";

    }

    private Optional<String> eventName(SyncEvent syncEvent) {
        try {
            return Optional.ofNullable(calendarWeb.getEvent(syncEvent.id()).getName());
        } catch (Exception e) {
            LOG.warn("didn't get event " + syncEvent.id(), e);
        }
        return Optional.empty();

    }
}
