package ru.yandex.qe.mail.meetings.ws.validation;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.JsonNode;

@JsonIgnoreProperties(ignoreUnknown = true)
public final class ValidationRequest {
    public final int id;
    public final String slug;
    public final String name;
    public final List<JsonNode> questions;

    @JsonCreator
    public ValidationRequest(
            @JsonProperty("id") int id,
            @JsonProperty("slug") String slug,
            @JsonProperty("name") String name,
            @JsonFormat(with = JsonFormat.Feature.ACCEPT_SINGLE_VALUE_AS_ARRAY)
            @JsonProperty("questions")
                    List<JsonNode> questions) {
        this.id = id;
        this.slug = slug;
        this.name = name;
        this.questions = questions;
    }

    @Nullable
    public String getAnswer(@Nonnull String questionSlug) {
        final JsonNode o = get(questionSlug);
        return convertJsonNodeToString(o);
    }

    private String convertJsonNodeToString(JsonNode o) {
        if (o != null) {
            if (o.isArray()) {
                StringBuilder result = new StringBuilder();
                for (JsonNode jsonNode : o) {
                    result.append(jsonNode.asText());
                    result.append(",");
                }
                if (result.length() > 1) {
                    result.setLength(result.length() - 1);
                }
                return result.toString();
            }
            return o.asText();
        }
        return null;
    }

    public Map<String, Object> toQuestionsMap() {
        if (checkSlulsInQuestions()) {
            return questions.stream().collect(
                    Collectors.toUnmodifiableMap(q -> q.get("slug").asText(), q -> convertJsonNodeToString(q.get("value"))));
        }
        Map<String, Object> result = new HashMap<>();
        for (JsonNode question : questions) {
            final Iterator<String> iterator = question.fieldNames();
            while (iterator.hasNext()) {
                final String key = iterator.next();
                result.put(key, convertJsonNodeToString(question.findValue(key)));
            }
        }
        return result;
    }

    private JsonNode get(String questionSlug) {
        if (checkSlulsInQuestions()) {
            for (JsonNode question : questions) {
                if (questionSlug.equalsIgnoreCase(question.get("slug").asText())) {
                    return question.get("value");
                }
            }
        } else {
            for (JsonNode question : questions) {
                final JsonNode value = question.findValue(questionSlug);
                if (value != null) {
                    return value;
                }
            }
        }
        return null;
    }

    private boolean checkSlulsInQuestions() {
        for (JsonNode question : questions) {
            if (question.get("slug") == null) {
                return false;
            }
        }
        return true;
    }

    @Override
    public String toString() {
        return "ValidationRequest{" +
                "id=" + id +
                ", slug='" + slug + '\'' +
                ", name='" + name + '\'' +
                ", questions=" + questions +
                '}';
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class Question {
        public final String slug;
        public final Object value;

        @JsonCreator
        public Question(
                @JsonProperty("slug") String slug,
                @JsonProperty("value") Object value) {
            this.slug = slug;
            this.value = value;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }

            Question question = (Question) o;

            if (slug != null ? !slug.equals(question.slug) : question.slug != null) {
                return false;
            }
            return value != null ? value.equals(question.value) : question.value == null;
        }

        @Override
        public int hashCode() {
            int result = slug != null ? slug.hashCode() : 0;
            result = 31 * result + (value != null ? value.hashCode() : 0);
            return result;
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        ValidationRequest that = (ValidationRequest) o;

        if (id != that.id) {
            return false;
        }
        if (slug != null ? !slug.equals(that.slug) : that.slug != null) {
            return false;
        }
        if (name != null ? !name.equals(that.name) : that.name != null) {
            return false;
        }
        return questions != null ? questions.equals(that.questions) : that.questions == null;
    }

    @Override
    public int hashCode() {
        int result = id;
        result = 31 * result + (slug != null ? slug.hashCode() : 0);
        result = 31 * result + (name != null ? name.hashCode() : 0);
        result = 31 * result + (questions != null ? questions.hashCode() : 0);
        return result;
    }
}
