package ru.yandex.qe.mail.meetings.ws.validation;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nonnull;

import com.fasterxml.jackson.annotation.JsonInclude;

@JsonInclude(JsonInclude.Include.NON_NULL)
public final class ValidationResult {
    public final static String INVALID_URL = "Invalid URL";

    public enum Status {
        OK,
        ERROR
    }

    private final Status status;
    private final Map<String, Set<String>> errors;

    public ValidationResult(Status status, String slug, Set<String> errors) {
        this.status = status;
        this.errors = Map.of(slug, errors);
    }

    private ValidationResult(Status status, Map<String, Set<String>> errors) {
        this.status = status;
        this.errors = errors;
    }

    public Status getStatus() {
        return status;
    }

    public Map<String, Set<String>> getErrors() {
        return errors.isEmpty() ? null : errors;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        ValidationResult that = (ValidationResult) o;
        return status == that.status &&
                Objects.equals(errors, that.errors);
    }

    @Override
    public int hashCode() {
        return Objects.hash(status, errors);
    }

    @Override
    public String toString() {
        return "ValidationResult{" +
                "status=" + status +
                ", errors=" + errors +
                '}';
    }

    public static ValidationResult merge(@Nonnull ValidationResult one, @Nonnull ValidationResult other) {
        Status status = one.status == Status.OK && other.status == Status.OK ? Status.OK : Status.ERROR;
        Map<String, Set<String>> errors = new HashMap<>(one.errors);
        other.errors.forEach((k, v) -> errors.computeIfAbsent(k, error -> new HashSet<>()).addAll(v));
        return new ValidationResult(status, Collections.unmodifiableMap(errors));
    }

    public static ValidationResult error(@Nonnull String slug) {
        return new ValidationResult(Status.ERROR, slug, Set.of("Incorrect value"));
    }

    public static ValidationResult error(@Nonnull String slug, String message) {
        return new ValidationResult(Status.ERROR, slug, Set.of(message));
    }

    public static ValidationResult error(@Nonnull String slug, Exception exception) {
        String message = exception.getMessage() == null ? "Unknown error" : exception.getMessage();
        return new ValidationResult(Status.ERROR, slug, Set.of(message));
    }

    public static ValidationResult success() {
        return new ValidationResult(Status.OK, Collections.emptyMap());
    }
}
