#pragma once

#include <string>
#include <boost/noncopyable.hpp>

#ifdef __clang__
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wsign-conversion"
#endif

#include <yplatform/ptree.h>

#ifdef __clang__
#pragma clang diagnostic pop
#endif

#include <mail_getter/content_type.h>
#include <mail_getter/alias_class_list.h>
#include <mail_getter/vdirect/keys_storage.h>
#include <mail_getter/service_factory.h>
#include <user_journal/service.h>
#include <ymod_tvm/module.h>
#include <mail/tvm_guard/ymod_tvm/ymod_tvm.h>
#include <mail/ymod_maildb/include/module.h>


#include <internal/sanitizer.h>
#include <internal/sanitizer_tykva.h>
#include <internal/logger.h>
#include <internal/fact_extractor/parser_config.h>
#include <internal/fact_extractor/fact_extractor.h>

namespace msg_body {

struct VdirectConfig {
    std::string uidScript;
    std::string smsScript;
    vdirect::KeysStorage keysStorage;

    bool isConfigured() const {
        return !(smsScript.empty() || uidScript.empty() || keysStorage.empty());
    }
};

struct MulcagateConfig {
    MulcagateConfig() {
        settings->retries = 1;
    }
    mail_getter::mulcagate::SettingsPtr settings = std::make_shared<mail_getter::mulcagate::Settings>();
};

struct MessageTextConfig {
    std::size_t maxPartLength = 0;
};

typedef boost::scoped_ptr<ContentTypeDetector> ContentTypeDetectorPtr;

class Configuration: private boost::noncopyable {
public:
    void load(const yplatform::ptree& tree, std::shared_ptr<ymod_tvm::tvm2_module> module);

    mail_getter::ServiceFactoryPtr mailStorage;
    user_journal::ServicePtr userJournalService;
    ymod_maildb::ModulePtr maildb;
    ContentTypeDetectorPtr contentTypeDetector;
    AliasClassList aliasClassList;
    ParserConfig factexParserConfig;
    FactExtractor::Config factexConfig;
    unsigned bigLettersTrimThreshold;
    VdirectConfig vdirect;
    SanitizerParams sanitizer;
    unsigned divTagDepthLimit;
    std::string webattachServer;
    std::chrono::hours inlineAttachesTTL;
    std::chrono::seconds inlineAttachesShieldTimeout;
    MessageTextConfig messageTextConfig;
    std::size_t lengthLimitAddress;
    std::shared_ptr<tvm_guard::Guard<ymod_tvm::tvm2_module>> tvmGuard;
    TvmGuardLoggerPtr tvmGuardLogger;
    Recognizer::WrapperPtr recognizer;
    std::vector<std::string> headersToPass;
    mail_getter::attach_sid::Keys keys;
    std::size_t coroutineStackSize;
    bool plainTextSanitize;
    std::atomic<sanitizer::Strategy> sanitizerStrategy;

private:
    typedef boost::property_tree::ptree ptree;

    void initExtVideoSingleton(const ptree& tree, const LogPtr& logger);
    void initContentTypeDetector(ContentTypeDetectorPtr& contentTypeDetector, const ptree& tree,
            const LogPtr& logger);
    void initFactExtractor(ParserConfig& factexParserConfig, FactExtractor::Config& factexConfig,
            const ptree& tree, const LogPtr& logger);
    void initJournalService(const ptree& tree);
    void initMacs();
    void initMailStorage(const ptree& tree);
    void initHeadersToPass(const ptree& tree);
};

class ConfigurationException: public std::runtime_error {
public:
    ConfigurationException(const std::string& msg)
        : std::runtime_error(msg)
    {}
};

}

YREFLECTION_ADAPT_ADT(msg_body::MulcagateConfig,
    YREFLECTION_WO_MEMBER_RENAMED(std::string, storage_name_space, settings->storageNameSpace)
    YREFLECTION_WO_MEMBER_RENAMED(std::string, service, settings->service)
    YREFLECTION_WO_MEMBER_RENAMED(unsigned, connect_timeout_ms, settings->connectTimeoutMs)
    YREFLECTION_WO_MEMBER_RENAMED(unsigned, get_timeout_ms, settings->getTimeoutMs)
    YREFLECTION_WO_MEMBER_RENAMED(unsigned, put_timeout_ms, settings->putTimeoutMs)
    YREFLECTION_WO_MEMBER_RENAMED(bool, keep_alive, settings->keepAlive)
)
