#ifndef DKIM_PARSER_H_2014_02_21
#define DKIM_PARSER_H_2014_02_21

#include <map>
#include <boost/algorithm/string/split.hpp>
#include <butil/split_key_value.h>
#include <message_body/dkim.h>

namespace msg_body {

class DkimParser {
    const std::string startLandmark;
    const std::string endLandmark;
    const std::string dkimName;
    const std::string headeriName;
    const std::string paramDelimiters;

    std::string getDkimHeaderPart(const std::string & header) const {
        const std::size_t start = header.find(startLandmark);
        if (start == std::string::npos) {
            return "";
        }
        const std::size_t end = header.find(endLandmark, start + startLandmark.size());
        return header.substr(start, end-start);
    }

    static Dkim::Status statusFromString(const std::string & status) {
        if(status=="pass") {
            return Dkim::valid;
        }
        if (status=="fail") {
            return Dkim::badsig;
        }
        return Dkim::unknown;
    }

    Dkim dkimFromString(const std::string & src) const {
        if (src.empty()) {
            return Dkim(Dkim::nosig);
        }
        std::vector<std::string> tokens;
        boost::split(tokens, src, boost::is_any_of(paramDelimiters));
        std::map<std::string,std::string> values;
        std::transform(tokens.begin(),tokens.end(),
                std::inserter(values, values.end()), ::splitKeyValue);
        return Dkim(statusFromString(values[dkimName]), values[headeriName]);
    }
public:
    DkimParser()
        : startLandmark("dkim="), endLandmark(";"), dkimName("dkim"),
          headeriName("header.i"), paramDelimiters(" \r\n\t")
    {}

    Dkim parse( const std::string & header ) {
        return dkimFromString(getDkimHeaderPart(header));
    }
};

}

#endif // DKIM_PARSER_H_2014_02_21
