#pragma once

#include <internal/pa_log.h>
#include <butil/http/error.h>
#include <butil/http/codes.h>
#include <butil/email/email.h>
#include <butil/email/helpers.h>
#include <internal/fact_extractor/fact_extractor.h>
#include <internal/fact_extractor/entities_to_extract_finder.h>
#include <internal/transformer_attributes.h>
#include <message_body/facts.h>
#include <internal/message_context.h>
#include <internal/yield_context.h>

namespace msg_body {

class MessagePartAsyncFactExtractor {
public:
    using RequestPtr = FactExtractor::RequestPtr;

    MessagePartAsyncFactExtractor(LogPtr logger, const ParserConfig& config,
            const FactExtractor::Config& extractorConfig,
            const std::string& requestId, YieldCtx yc, const GetClusterClient& getClusterClient)
            : logger_(logger)
            , config_(config)
            , extractorConfig_(extractorConfig)
            , requestId_(requestId)
            , yc_(yc)
            , getClusterClient_(getClusterClient) {
    }

    virtual ~MessagePartAsyncFactExtractor() {}

    RequestPtr extractFactsBegin(const MessagePart& messagePart,
            const MessageContext& messageContext,
            const TransformerAttributes& attrs) const {
        if (!attrs.flags.showContentMeta) {
            return RequestPtr();
        }
        try {
            const Entities entities = findEntitiesToExtract(messageContext, attrs);
            if (entities.empty()) {
                return RequestPtr();
            }
            return extractFacts(entities, messagePart, messageContext, attrs);
        } catch (const std::exception& e) {
            const std::string host = http::url(extractorConfig_.url).server();
            MBODY_LOG_ERROR(logger(), log::where_name="fact_extractor", log::message="host=" + host, log::exception=e);
        }
        return RequestPtr();
    }

    FactsPtr extractFactsEnd(const RequestPtr& request, const PaLog& paLog) const {
        if (!request.get()) {
            return FactsPtr();
        }
        const std::string host = http::url(extractorConfig_.url).server();
        try {
            const auto ctx = boost::make_shared<MessageBodyTaskContext>(requestId_);
            const auto response = http_getter::asyncRun(*getClusterClient_("fact_extractor_client"), ctx, *request, yc_);
            paLog.write();
            if (http::codeIsOk(response.status)) {
                return boost::make_shared<std::string>(response.body);
            } else {
                MBODY_LOG_ERROR(logger(), log::where_name="fact_extractor", log::message="host=" + host + ", status=" + std::to_string(response.status));
            }
        } catch (const std::exception& e) {
            MBODY_LOG_ERROR(logger(), log::where_name="fact_extractor", log::message="host=" + host, log::exception=e);
        }
        return FactsPtr();
    }

protected:
    typedef FactExtractor::Query::Entities Entities;

    struct QueryAttributes {
        std::time_t time;
        long timeZone;
        std::string email;
        std::string domain;
        std::string uid;
        std::string mid;
        std::string types;
        std::string part;
        std::string originalMessageId;
        std::string lang;
    };

    LogPtr logger() const {
        return logger_;
    }

    const ParserConfig& config() const {
        return config_;
    }

    RequestPtr extractFacts(const Entities& entities,
            const MessagePart& messagePart,
            const MessageContext& messageContext,
            const TransformerAttributes& attrs) const {
        if (extractorConfig_.url.empty()) {
            return RequestPtr();
        }

        QueryAttributes queryAttributes;
        getQueryAttributes(queryAttributes, messageContext, attrs);

        const std::string content = getContent(messagePart);
        if (content.empty()) {
            return RequestPtr();
        }

        FactExtractor service(extractorConfig_);
        return buildQuery(service, content, entities, queryAttributes).request(requestId_);
    }

    FactExtractor::Query buildQuery(const FactExtractor& service, const std::string& content,
            const Entities& entities, const QueryAttributes& queryAttributes) const {
        FactExtractor::Query query = service.extract(content)
                .entities(entities)
                .part(queryAttributes.part)
                .domain(queryAttributes.domain)
                .time(queryAttributes.time)
                .timeZone(queryAttributes.timeZone)
                .mid(queryAttributes.mid)
                .originalMessageId(queryAttributes.originalMessageId)
                .types(queryAttributes.types)
                .lang(queryAttributes.lang);
        if (!(queryAttributes.email.empty() || queryAttributes.uid.empty())) {
            query.contact (queryAttributes.email, queryAttributes.uid);
        }
        return query;
    }

    void getQueryAttributes(QueryAttributes &queryAttributes,
            const MessageContext& messageContext,
            const TransformerAttributes& attrs) const {
        std::string login;

        queryAttributes.time = attrs.messageDate;
        queryAttributes.timeZone = attrs.timeZoneOffset;
        queryAttributes.email = attrs.from;
        split_domain(attrs.from, login, queryAttributes.domain);

        queryAttributes.mid = attrs.mid;
        queryAttributes.uid = attrs.uid;
        queryAttributes.types = attrs.typesString();
        queryAttributes.part = getPartAttribute();
        queryAttributes.originalMessageId = messageContext.originalMessageId_;
        queryAttributes.lang = attrs.lang;
    }

    virtual Entities findEntitiesToExtract(const MessageContext& messageContext,
        const TransformerAttributes& attrs) const = 0;

    virtual const std::string getContent(const MessagePart& messagePart) const = 0;

    virtual const std::string getPartAttribute() const = 0;

private:
    LogPtr logger_;
    const ParserConfig& config_;
    const FactExtractor::Config& extractorConfig_;
    const std::string requestId_;
    YieldCtx yc_;
    GetClusterClient getClusterClient_;
};

class TextAsyncFactExtractor : public MessagePartAsyncFactExtractor {
public:
    TextAsyncFactExtractor(LogPtr logger, const ParserConfig& config,
            const FactExtractor::Config& extractorConfig,
            const std::string& requestId, YieldCtx yc, const GetClusterClient& getClusterClient)
            : MessagePartAsyncFactExtractor(logger, config, extractorConfig, requestId, yc, getClusterClient) {
    }

    virtual ~TextAsyncFactExtractor() {}

protected:
    virtual Entities findEntitiesToExtract(const MessageContext& messageContext,
            const TransformerAttributes& attrs) const {
        Entities entities = TextEntitiesToExtractFinder(attrs.types).find(config());

        if(messageContext.listUnsubscribe_) {
            Entities::const_iterator searchForUnsubscribe = entities.find("unsubscribe");
            if(searchForUnsubscribe != entities.end()) {
                entities.erase(searchForUnsubscribe);
            }
        }

        return entities;
    }

    virtual const std::string getContent(const MessagePart& messagePart) const {
        return messagePart.content;
    }

    virtual const std::string getPartAttribute() const {
        return "message";
    }
};

} // namespace fact_extractor
