#ifndef ENTITIES_TO_EXTRACT_FINDER_H_2013_12_18
#define ENTITIES_TO_EXTRACT_FINDER_H_2013_12_18

#include <boost/algorithm/cxx11/any_of.hpp>
#include <boost/algorithm/cxx11/all_of.hpp>
#include <boost/range/adaptor/filtered.hpp>
#include <boost/range/algorithm/transform.hpp>
#include <boost/range/algorithm/for_each.hpp>

#include <butil/email/email.h>
#include <mail_getter/MessageAccessDefinitions.h>
#include <internal/fact_extractor/parser_config.h>
#include <internal/fact_extractor/fact_extractor.h>
#include <internal/message_tree.h>

namespace msg_body {

class EntitiesToExtractFinder {
public:
    typedef std::set<int> MessageTypes;
    typedef Query::Entities Entities;

    EntitiesToExtractFinder(const MessageTypes &messageTypes)
    : messageTypes_(messageTypes) {}
    virtual ~EntitiesToExtractFinder() {}

    Entities find(const ParserConfig& config) const {
        Entities entities;
        boost::for_each(config.rules | boost::adaptors::filtered(boost::bind(
                &EntitiesToExtractFinder::satisfiesRule, this, _1)),
                boost::bind(EntitiesToExtractFinder::addEntitiesFromRule, _1, boost::ref(entities)));

        return entities;
    }

protected:
    virtual bool satisfiesRule(const ParserConfig::Rule &rule) const = 0;

    static void addEntitiesFromRule(const ParserConfig::Rule &rule, Entities &entities) {
        entities.insert(rule.entities.begin(), rule.entities.end());
    }

    bool satisfiesIncludeRestriction(const ParserConfig::Rule::Types &include) const {
        return std::includes(messageTypes_.begin(), messageTypes_.end(),
                include.begin(), include.end());
    }

    bool satisfiesExcludeRestriction(const ParserConfig::Rule::Types &exclude)  const {
        return std::find_first_of(messageTypes_.begin(), messageTypes_.end(),
                exclude.begin(), exclude.end()) == messageTypes_.end();
    }

private:
    const MessageTypes & messageTypes_;
};

class TextEntitiesToExtractFinder : public EntitiesToExtractFinder {
public:
    TextEntitiesToExtractFinder(const MessageTypes & messageTypes)
    : EntitiesToExtractFinder(messageTypes) {}

    virtual ~TextEntitiesToExtractFinder() {};

protected:
    virtual bool satisfiesRule(const ParserConfig::Rule &rule) const {
        return satisfiesIncludeRestriction(rule.include)
            && satisfiesExcludeRestriction(rule.exclude);
    }
};

} // namespace fact_extractor

#endif /* ENTITIES_TO_EXTRACT_FINDER_H_2013_12_18 */
