#pragma once

#include <set>
#include <string>
#include <map>
#include <iostream>
#include <boost/lexical_cast.hpp>
#include <boost/algorithm/string/join.hpp>
#include <boost/range/adaptor/transformed.hpp>
#include <internal/fact_extractor/transport.h>

namespace msg_body {

inline std::string joinKeyValue(const std::pair<const std::string,std::string> & p) {
    return p.first + "=" + p.second;
}

class Query {
public:
    typedef std::set<std::string> Entities;
    typedef std::map<std::string, std::string> Arguments;
    typedef Transport::RequestPtr RequestPtr;

    Query(const std::string& content, const Transport& transport)
        : content_(content)
        , transport_(transport)
    {
        if (content.empty()) {
            throw std::invalid_argument("Fact extractor content is empty");
        }
    }

    Query & entities(const Entities & entities) {
        entities_ = entities;
        return *this;
    }

    Query & part(const std::string & part) {
        addArgument("part", part);
        return *this;
    }

    Query & domain(const std::string & domain) {
        addArgument("domain", domain);
        return *this;
    }

    Query & contact(const std::string & email, const std::string & uid) {
        addArgument("email", email);
        addArgument("uid", uid);
        return *this;
    }

    Query & types(const std::string & types) {
        if (!types.empty()) {
            addArgument("types", types);
        }
        return *this;
    }

    Query & originalMessageId(const std::string & messageId) {
        if (!messageId.empty()) {
            addArgument("original_message_id", messageId);
        }
        return *this;
    }

    Query & mid(const std::string& mid) {
        addArgument("mid", mid);
        return *this;
    }

    Query & time(std::time_t time) {
        addArgument("time", boost::lexical_cast<std::string>(time));
        return *this;
    }

    Query & timeZone(long timeZone) {
        addArgument("time_zone", boost::lexical_cast<std::string>(timeZone));
        return *this;
    }

    Query & lang(const std::string& lang) {
        if (!lang.empty()) {
            addArgument("lang", lang);
        }
        return *this;
    }

    RequestPtr request(const std::string& requestId) const {
        if( entities_.empty() ) {
            throw std::logic_error("Fact extractor query is empty");
        }
        return transport().post(getTransportParams(), content(), requestId);
    }

private:
    const std::string & content() const {
        return content_;
    }

    const Transport & transport() const {
        return transport_;
    }

    std::string getTransportParams() const {
        using boost::adaptors::transformed;
        using boost::join;
        return std::string("/?e=") + join( entities_, ",") + '&' +
            join( arguments_ | transformed(&joinKeyValue), "&");
    }

    void addArgument( const std::string & name, const std::string & value ) {
        if (value.empty()) {
            throw std::invalid_argument("Fact extractor argument " + name + " is empty!");
        }
        arguments_[name] = value;
    }

    const std::string & content_;
    const Transport & transport_;
    Entities entities_;
    Arguments arguments_;
};

}
