#pragma once

#include <string>
#include <boost/lexical_cast.hpp>
#include <internal/http_call.h>

namespace msg_body {

class Transport {
public:
    using RequestPtr = std::shared_ptr<http_getter::Request>;
    virtual RequestPtr post(const std::string & params, const std::string & content, const std::string& requestId) const = 0;

    virtual ~Transport() {}
};

class TransportImpl : public Transport {
public:
    struct Config {
        std::string url;
        unsigned timeout;
        std::chrono::milliseconds connectTimeout;
        bool keepAlive;
        bool logPostBody;
        Config( const std::string & url, unsigned timeout, unsigned connectTimeout, bool keepAlive, bool logPostBody )
            : url(url), timeout(timeout), connectTimeout(connectTimeout), keepAlive(keepAlive), logPostBody(logPostBody)
        {}
        Config()
            : timeout(0), connectTimeout(1), keepAlive(false), logPostBody(false)
        {}
    };

    TransportImpl( const Config & config )
        : config(config)
    {
        if (config.url.empty()) {
            throw std::invalid_argument("FactExtract: url is empty");
        }
    }

    virtual RequestPtr post(const std::string & params, const std::string & content, const std::string& requestId) const {
        using namespace http_getter;
        return std::make_shared<Request>(http_getter::post(config.url + params)
                                        .headers(http_getter::requestId=requestId, "Content-Length"_hdr=std::to_string(content.size()), "Content-Type"_hdr="text/html")
                                        .body(content)
                                        .logPostArgs(config.logPostBody)
                                        .keepAlive(config.keepAlive)
                                        .timeouts(toYplatformDuration(std::chrono::microseconds(config.timeout)), toYplatformDuration(std::chrono::microseconds(config.connectTimeout)))
                                        .make());
    }
private:
    Config config;
};
} // namespace msg_body

