#ifndef FACT_EXTRACTOR_XML_PARSER_H_2013_12_18
#define FACT_EXTRACTOR_XML_PARSER_H_2013_12_18

#include "parser_config.h"
#include <boost/property_tree/ptree.hpp>
#include <boost/property_tree/xml_parser.hpp>
#include <boost/range/adaptor/filtered.hpp>
#include <boost/range/algorithm/copy.hpp>
#include <boost/range/algorithm/for_each.hpp>
#include <boost/range/algorithm/transform.hpp>
#include <boost/bind.hpp>
#include <boost/lambda/bind.hpp>
#include <boost/lambda/construct.hpp>
#include <boost/lexical_cast.hpp>
#include <algorithm>
#include <iterator>

namespace msg_body {

namespace xml {

typedef boost::property_tree::ptree Tree;
typedef Tree::const_iterator::reference Node;
typedef Tree::const_iterator::pointer NodePtr;

class XmlGetterBase {
    NodePtr ptreeNode_;
    static const Tree emptyTree;
public:
    bool empty() const {
        return !ptreeNode_;
    }

    const Tree & node() const {
        return ptreeNode_ ? ptreeNode_->second : emptyTree;
    }

    const Tree & getChild( const std::string & name ) const {
        return node().get_child(name, emptyTree);
    }

    std::string get( const std::string & name ) const {
        return node().get<std::string>(name);
    }

    std::string value() const {
        return node().get_value<std::string>();
    }

    const std::string & nodeName() const {
        if (ptreeNode_) {
            return ptreeNode_->first;
        } else {
            throw std::runtime_error("logical error - XmlGetterBase::nodeName() called for empty node!");
        }
    }

    template <class Out>
    Out getChildNode( const std::string & which ) const {
        Tree::const_assoc_iterator i( node().find(which) );
        if ( i == node().not_found() ) {
            return Out();
        }
        return Out( *(node().to_iterator(i)) );
    }

    XmlGetterBase() : ptreeNode_(0) {}
    XmlGetterBase( Node ptreeNode ) : ptreeNode_(&ptreeNode) {}

    XmlGetterBase & operator = ( Node ptreeNode ) {
        ptreeNode_ = &ptreeNode;
        return *this;
    }
};

template <typename Pred, typename Out>
inline Out transfomChildNodes(const Tree & node, Pred p, Out out) {
    using namespace boost::lambda;
    boost::copy( node | boost::adaptors::filtered(p), out );
    return out;
}

class Type : protected XmlGetterBase {
public:
    typedef msg_body::TypeMsg::Type Value;
    Value get() const {
        return Value(node().get_value<int>());
    }

    operator Value() const {
        return get();
    }

    Type( Node ptreeNode ) : XmlGetterBase(ptreeNode) {}
};

class Types : protected XmlGetterBase {
public:
    typedef std::vector<Type> Value;
    Value get() const {
        Value out;
        transfomChildNodes(node(), boost::bind(Types::nodeIsType, _1),
                std::back_inserter(out));
        return out;
    }

    operator Value () const {
        return get();
    }

    static bool nodeIsType( Node node ) {
        return node.first == "type";
    }

    Types() {}

    Types( Node ptreeNode ) : XmlGetterBase(ptreeNode) {}
};

class Id : protected XmlGetterBase {
public:
    typedef std::string Value;
    Value get() const {
        return value();
    }

    operator Value() const {
        return get();
    }

    Id( Node ptreeNode ) : XmlGetterBase(ptreeNode) {}
};

typedef std::vector<Id> Ids;

class TypeMsg : protected XmlGetterBase {
public:
    Ids ids() const {
        Ids out;
        transfomChildNodes(node(), boost::bind(TypeMsg::nodeIsId, _1),
                std::back_inserter(out));
        return out;
    }

    Types include() const {
        return getChildNode<Types>("include");
    }

    Types exclude() const {
        return getChildNode<Types>("exclude");
    }

    TypeMsg( Node ptreeNode ) : XmlGetterBase(ptreeNode) {}

    static bool nodeIsId( Node node ) {
        return node.first == "id";
    }
};//class TypeMsg


class GetTypeMsgs {
public:
    typedef const Tree & argument_type;
    typedef std::vector<TypeMsg> result_type;

    result_type operator() (argument_type arg) const {
        result_type out;
        transfomChildNodes( arg, boost::bind(GetTypeMsgs::nodeIsTypeMsg, _1),
                std::back_inserter(out));
        return out;
    }

    static bool nodeIsTypeMsg( Node node ) {
        return node.first == "type-msg";
    }
};

struct ConvertTypeMsg {
    typedef msg_body::TypeMsg result_type;
    typedef const msg_body::xml::TypeMsg argument_type;

    template <typename Out>
    void getTypes( const Types & types, Out out ) const {
        const Types::Value values(types);
        std::copy(values.begin(), values.end(), out);
    }

    template <typename Out>
    void getIds( const Ids & ids, Out out ) const {
        std::copy(ids.begin(), ids.end(), out);
    }

    result_type operator() ( argument_type & arg ) const {
        result_type retval;
        getIds( arg.ids(), std::inserter(retval.entities, retval.entities.end()) );
        getTypes( arg.include(), std::back_inserter(retval.include) );
        getTypes( arg.exclude(), std::back_inserter(retval.exclude) );
        return retval;
    }

};

} //namespace xml
} //namespace msg_body


#endif /* FACT_EXTRACTOR_XML_PARSER_H_2013_12_18 */
