#pragma once

#include <chrono>
#include <http_getter/http_request.h>
#include <ymod_httpclient/call.h>
#include <ymod_tvm/module.h>

namespace msg_body {

struct MessageBodyTaskContext: public yplatform::task_context {
    MessageBodyTaskContext(const std::string& requestId)
        : yplatform::task_context(requestId) {
    }

    virtual const std::string& get_name() const {
        static const std::string name = "message_body";
        return name;
    }
};

using Microsecs = std::chrono::microseconds;

template<class T, class L>
inline auto toYplatformDuration(const std::chrono::duration<T, L>& time) {
    return yplatform::time_traits::duration_cast<yplatform::time_traits::duration>(time);
};

using GetHttpClient = std::function<std::shared_ptr<yhttp::simple_call> ()>;
inline GetHttpClient makeGetHttpClient() {
    return [] { return yplatform::find<yhttp::simple_call, std::shared_ptr>("http"); };
}

using GetClusterClient = std::function<std::shared_ptr<ymod_httpclient::cluster_call> (const std::string&)>;
inline GetClusterClient makeGetClusterClient() {
    return [] (const std::string& client_name) { return yplatform::find<ymod_httpclient::cluster_call, std::shared_ptr>(client_name); };
}

using GetServiceTicket = std::function<std::string(const std::string& service)>;
using Tvm2ModulePtr = std::shared_ptr<ymod_tvm::tvm2_module>;

inline auto makeGetServiceTicket(Tvm2ModulePtr tvm) {
    return [tvm] (const std::string& service) {
        std::string ticket;
        if (auto ec = tvm->get_service_ticket(service, ticket)) {
            throw std::runtime_error("Http request to " +  service + " failed: Cannot get service ticket");
        }
        return ticket;
    };
}

inline Tvm2ModulePtr getTvmModule() {
    return yplatform::find<ymod_tvm::tvm2_module, std::shared_ptr>("ymod_tvm");
}

inline std::string getMulcagateServiceTicket() {
    std::string serviceTicket;
    auto err = getTvmModule()->get_service_ticket("mulcagate", serviceTicket);
    if (err) {
        throw std::runtime_error("Failed to get mucagate service ticket: " + err.message());
    }
    return serviceTicket;
}

}
