#pragma once

#include <string>
#include <chrono>
#include <butil/http/arguments.h>
#include <internal/http_call.h>
#include <mail_getter/sanitizer/sanitizer_response_processor.h>
#include <internal/pa_log.h>
#include <butil/http/codes.h>
#include <internal/yield_context.h>

namespace msg_body {

struct SanitizerParams {
    std::string url;
    std::string doProxy;
    std::string methodHttp;
    std::string methodHttps;
    std::string methodSpam;
    std::chrono::milliseconds connectTimeout{1000};
    std::chrono::milliseconds totalTimeout{1000};
    bool keepAlive = false;
    std::int8_t max_retries;
    bool logPostBody = false;

    bool valid() const {
        return !url.empty()
            && !methodHttp.empty()
            && !methodHttps.empty()
            && !methodSpam.empty();
    }
};

class Sanitizer {
public:
    Sanitizer(const SanitizerParams& params, bool isSecure, bool isSpam,
              const std::string& requestId, const http::headers& headersToPass,
              GetServiceTicket&& getServiceTicket_, YieldCtx yc, const GetClusterClient& getClusterClient_);

    inline mail_getter::SanitizerParsedResponse sanitizeEnd(
                    const std::string& content,
                    const std::string& uid,
                    const std::string& mid,
                    const std::string& stid,
                    const std::string& contentType,
                    const std::optional<std::string>& from,
                    const std::optional<std::string>& system,
                    const PaLog& paLog) const;

private:
    std::string genRequestId() const;
    std::string getSanitizeMethod(
            const SanitizerParams& params,
            bool isSecure,
            bool isSpam) const;

    SanitizerParams params_;
    bool isSecure_;
    bool isSpam_;
    std::string requestId_;
    http::headers headersToPass_;
    YieldCtx yc_;
    GetServiceTicket getServiceTicket_;
    GetClusterClient getClusterClient_;
};

std::string getHeaderI(const std::string& name, const ymod_httpclient::headers_dict &headers);
inline mail_getter::SanitizerParsedResponse Sanitizer::sanitizeEnd(
                const std::string& content,
                const std::string& uid,
                const std::string& mid,
                const std::string& stid,
                const std::string& contentType,
                const std::optional<std::string>& from,
                const std::optional<std::string>& system,
                const PaLog& paLog) const {
    using namespace http_getter;
    const std::string method  = getSanitizeMethod(params_, isSecure_, isSpam_);
    std::string ticket = getServiceTicket_("sanitizer");
    const Request request = post(params_.url)
                            .getArgs(
                                "s"_arg=method,
                                "uid"_arg=uid,
                                "mid"_arg=mid,
                                "stid"_arg=stid,
                                "mimetype"_arg=contentType,
                                "from"_arg=from,
                                "system"_arg=system,
                                "id"_arg=requestId_)
                            .headers(serviceTicket=ticket, requestId=requestId_, "hdrs"_hdr=headersToPass_)
                            .body(content)
                            .logPostArgs(params_.logPostBody)
                            .keepAlive(params_.keepAlive)
                            .timeouts(params_.totalTimeout, params_.connectTimeout)
                            .make();

    try {
        ymod_httpclient::response response;
        const auto ctx = boost::make_shared<MessageBodyTaskContext>(requestId_);
        response = asyncRun(*getClusterClient_("sanitizer_client"), ctx, request, yc_);
        if (::http::codeIsOk(response.status)) {
            paLog.write();
            return mail_getter::processSanitizerResponse(response.body, getHeaderI("Content-Type", response.headers));
        }
        throw std::runtime_error("Sanitizer returns HTTP code " + std::to_string(response.status));
    } catch (const std::runtime_error& e) {
        throw std::runtime_error("Http request to sanitizer failed: " + std::string(e.what()));
    }
}

}
