#pragma once

#ifdef __clang__
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wsign-conversion"
#endif
#include <ymod_webserver/response.h>
#ifdef __clang__
#pragma clang diagnostic pop
#endif

#include <internal/logger.h>
#include <internal/exceptions.h>
#include <butil/network/real_ip_from_headers.h>

#include <optional>

namespace msg_body {

class RequestContext {
public:
    RequestContext(const ymod_webserver::http::stream_ptr& stream)
        : stream_(stream)
        , logger_(std::make_shared<ContextLogger>(makeLoggerWithRequestId(requestId())))
    {}

    void responseSuccess(const std::string& str) {
        response(str, ymod_webserver::codes::ok);
    }

    void responseInternalError(const std::string& str) {
        response(str, ymod_webserver::codes::internal_server_error);
    }

    void responseInvalidRequest(const std::string& str) {
        response(str, ymod_webserver::codes::bad_request);
    }

    void rejectByTvmTicket() {
        response("", ymod_webserver::codes::unauthorized);
    }

    std::string getRequiredArg(const std::string& name) const {
        const auto iter = stream_->request()->url.params.find(name);
        if (iter == stream_->request()->url.params.end()) {
            throw ParamsException("parameter not found: " + name);
        }
        return iter->second;
    }

    std::string getRequiredNonEmptyArg(const std::string& name) const {
        const std::string res = getRequiredArg(name);
        if (res.empty()) {
            throw ParamsException("parameter is empty: " + name);
        }
        return res;
    }

    std::string getArg(const std::string& name) const {
        const auto iter = stream_->request()->url.params.find(name);
        return iter == stream_->request()->url.params.end() ? "" : iter->second;
    }

    std::optional<std::string> getOptionalArg(const std::string& name) const {
        const auto iter = stream_->request()->url.params.find(name);
        return iter == stream_->request()->url.params.end() ? std::nullopt : std::make_optional(iter->second);
    }

    std::optional<std::string> getOptionalHeader(const std::string& key) const {
        const std::string n = boost::algorithm::to_lower_copy(key);
        const auto& headers = stream_->request()->headers;
        const auto iter = headers.find(n);
        return iter == headers.end() ? std::nullopt : std::make_optional(iter->second);
    }

    std::string endpoint() const {
        return stream_->request()->url.make_full_path();
    }

    std::vector<std::string> getRequiredArgs(const std::string& name) const {
        std::vector<std::string> res;
        auto values_range = stream_->request()->url.params.equal_range(name);
        std::for_each(values_range.first, values_range.second, [&](auto it) {
                        if (!it.second.empty()) {
                            res.emplace_back(std::move(it.second));
                        }});
        if (res.empty()) {
            throw ParamsException("parameter is empty: " + name);
        }
        return res;
    }

    std::string getHeader(const std::string& key) const {
        const auto& headers = stream_->request()->headers;
        const auto iter = headers.find(key);
        return iter == headers.end() ? "" : iter->second;
    }

    std::string requestId() const {
        return getHeader("x-request-id");
    }

    LogPtr logger() const {
        return logger_;
    }

    std::string remoteIp() const {
        return real_ip::getFromHeaders(getOptionalHeader("X-Forwarded-For"), getOptionalHeader("X-Real-Ip"),
                                       stream_->ctx()->remote_address);
    }

private:
    void response(const std::string& str, ymod_webserver::codes::code code) {
        stream_->set_code(code);
        stream_->set_content_type("application/json");
        stream_->result_body(str + "\r\n");
    }

    ymod_webserver::http::stream_ptr stream_;
    LogPtr logger_;
};

} // namespace msg_body
