#pragma once

#include <boost/filesystem.hpp>
#include <boost/algorithm/string.hpp>
#include <butil/strtools.h>
#include <yamail/data/reflection/reflection.h>
#include <mail_getter/MessageAccessDefinitions.h>
#include <mail_getter/content_type.h>
#include <message_body/embed_info.h>

namespace msg_body {

typedef std::unique_ptr<MetaAttributes> MetaAttributesPtr;
typedef std::unique_ptr<MetaLevel> MetaLevelPtr;

struct MessagePart {
    std::string hid;
    std::string stid;
    std::string realName;
    std::string content;
    std::string metaType;
    std::string textType;
    bool isPhishing = false;
    bool isMain = false;
    bool isAttach = false;
    bool isMixedAttach = true;
    bool isComplete = true;
    bool divLimitExceeded = false;
    bool afterTykva = false;
    MimeType contentType;
    EmbedInfos videoLinks;
    MetaLevel bodyStruct;
    MetaPart headerStruct;
    MetaAttributes messageHeader;
    std::optional<unsigned> length;
};

inline bool isNarodAttachment(const MessagePart& part) {
    static const std::string narodPattern = "narod_attachment_links.html";
    if (part.headerStruct.name() != narodPattern || part.headerStruct.fileName() != narodPattern) {
        return false;
    }
    return true;
}

inline bool isPkpassAttachment(const MessagePart& part) {
    if (part.contentType == MimeType("application", "vnd.apple.pkpass")) {
        return true;
    }
    const std::string& filename = part.headerStruct.name();
    const std::string extension = boost::filesystem::path(filename).extension().string();
    return tolow(extension) == ".pkpass";
}

inline std::string classifyMetatype(const MessagePart& part) {
    if (isNarodAttachment(part)) {
        return "narod";
    } else if (isPkpassAttachment(part)) {
        return "pkpass";
    }
    const MimeType& mimeType = part.contentType;
    if (mimeType == MimeType("text", "plain")) {
        return "text";
    } else if (mimeType == MimeType("text", "html")) {
        return "text";
    } else if (mimeType == MimeType("text", "enriched")) {
        return "text";
    } else if (mimeType == MimeType("text", "calendar")) {
        return "calendar";
    } else if (mimeType == MimeType("message", "delivery-status")) {
        return "text";
    } else {
        return "binary";
    }
}

inline bool isWindat(const MessagePart& part) {
    return (boost::algorithm::to_lower_copy(part.headerStruct.name()) == "winmail.dat" ||
                part.contentType.subtype() == "ms-tnef");
}

}

