#include <boost/property_tree/xml_parser.hpp>
#include <pa/async.h>

#include <yplatform/ptree.h>

#include <user_journal/service_factory.h>
#include <internal/ext_video.h>
#include <internal/external_transformers_selector.h>
#include <internal/phishing.h>
#include <internal/config.h>
#include <internal/macs/logging.h>
#include <internal/mulcagate/http.h>
#include <mail_getter/mulcagate/integration.h>
#include <mail_getter/attach_sid_keys.h>
#include <yamail/data/deserialization/ptree.h>
#include <ymod_tvm/module.h>
#include <tvm_guard/tvm_guard.h>
#include <limits>
#include <yamail/data/reflection/reflection.h>

namespace msg_body {

using boost::property_tree::ptree;

void badAllocHandler() {
    const auto logger = std::make_shared<ContextLogger>(makeLoggerWithRequestId(""));
    MBODY_LOG_ERROR(logger, log::message="global bad_alloc handler, will abort");
    abort();
}

void Configuration::initMacs() {
    maildb = yplatform::find<ymod_maildb::Module, std::shared_ptr>("maildb");
}

void Configuration::initMailStorage(const ptree& tree) {
    using namespace mail_getter;

    auto httpClient = getMulcagateHttpClientAdaptor();
    const auto& mulcagateTree = tree.get_child("mulcagate");
    auto mulcagateConfig = yamail::data::deserialization::fromPtree<MulcagateConfig>(mulcagateTree);

    mailStorage = createServiceFactory(std::move(mulcagateConfig.settings), std::move(httpClient));
}

void Configuration::initContentTypeDetector(ContentTypeDetectorPtr& contentTypeDetector,
        const ptree& tree, const LogPtr& logger) {
    const std::string mimeFilePath = tree.get("content_type_detector.mime_types", "");
    const std::string issuesFilePath = tree.get("content_type_detector.libmagic_issues", "");
    const std::string magicFilePath = tree.get("content_type_detector.magic_file", "");
    try {
        contentTypeDetector.reset(new ContentTypeDetector(mimeFilePath, issuesFilePath, magicFilePath));
    } catch (const std::exception& e) {
        MBODY_LOG_ERROR(logger, log::message="exception while initializating content type detector", log::exception=e);
    }
}

void Configuration::initFactExtractor(ParserConfig& factexParserConfig,
        FactExtractor::Config& factexConfig, const ptree& tree, const LogPtr& logger) {
    factexParserConfig = createParserConfig(tree.get("factex.parser_config", ""), logger);
    const std::string factexUrl = tree.get("factex.url", "");
    const double factexTimeout = tree.get<double>("factex.timeout", 3) * 1000 * 1000;
    const double factexConnectTimeout = tree.get<double>("factex.connect_timeout", 3) * 1000 * 1000;
    const bool factexKeepAlive = tree.get<bool>("factex.keep_alive");
    const bool factexLogPostBody = tree.get<bool>("factex.post_args_logging");
    factexConfig = FactExtractor::Config(factexUrl, static_cast<unsigned int>(factexTimeout), static_cast<unsigned int>(factexConnectTimeout), factexKeepAlive, factexLogPostBody);
}

void initVdirect(VdirectConfig& vdirect, const ptree& tree) {
    vdirect.smsScript = tree.get("vdirect.sms_script", "");
    vdirect.uidScript = tree.get("vdirect.uid_script", "");

    std::ifstream input(tree.get<std::string>("vdirect.keys_path"));
    vdirect.keysStorage = vdirect::KeysStorage(input);
}

void initRecognizer(Recognizer::WrapperPtr& recognizer, const ptree& tree) {
    std::string languageDict = tree.get<std::string>("recognizer.language_dict");
    std::string languageWeights = tree.get<std::string>("recognizer.language_weights");
    std::string encodingDict = tree.get<std::string>("recognizer.encoding_dict");
    recognizer = Recognizer::create(languageDict.c_str(), languageWeights.c_str(), encodingDict.c_str());
}

void initSanitizer(SanitizerParams& sanitizer, const ptree& tree) {
    sanitizer.url = tree.get("sanitizer.url", "");
    sanitizer.doProxy = tree.get("sanitizer.do_proxy", "0");
    sanitizer.methodHttp = tree.get("sanitizer.method_http", "");
    sanitizer.methodHttps = tree.get("sanitizer.method_https", "");
    sanitizer.methodSpam = tree.get("sanitizer.method_spam", "");
    sanitizer.connectTimeout = std::chrono::milliseconds(tree.get<unsigned>("sanitizer.connect_timeout_ms"));
    sanitizer.totalTimeout = std::chrono::milliseconds(tree.get<unsigned>("sanitizer.total_timeout_ms"));
    sanitizer.keepAlive = tree.get<bool>("sanitizer.keep_alive");
    sanitizer.max_retries =  tree.get<std::int8_t>("sanitizer.max_retries", 0);
    sanitizer.logPostBody = tree.get<bool>("sanitizer.post_args_logging");
    if (!sanitizer.valid()) {
        throw ConfigurationException("Failed to init sanitizer");
    }
}

void Configuration::initExtVideoSingleton(const ptree& tree, const LogPtr& logger) {
    ExtVideoSingleton extVideo;
    const std::string extVideoPath = tree.get("ext_video", "");
    try {
        extVideo.instance().loadHostings(extVideoPath);
    } catch (std::runtime_error& e) {
        MBODY_LOG_ERROR(logger, log::message="load ext video failed, path=" + extVideoPath, log::exception=e);
    }
}

void initExternalTransformersSelector(const ptree& tree) {
    const std::string ext_trans_cfg = tree.get("external_transformers", "");
    ExternalTransformersSelector::init(ext_trans_cfg);
}

void initPhishing(const ptree& tree) {
    PhishingChecker& phishingChecker = PhishingSingleton().instance();
    const std::string regexpFile = tree.get("phishing.regexp", "");
    phishingChecker.initRegular(regexpFile);

    const std::string patternFile = tree.get("phishing.pattern", "");
    const std::string wumanberStatus = tree.get("phishing.wumanber", "");
    if ("on" == wumanberStatus) {
        phishingChecker.loadPatternFrom(patternFile);
    }
}

void initDariaViewMaker(std::size_t& lengthLimitAddress, const ptree& tree) {
    lengthLimitAddress = tree.get<std::size_t>("daria.length_limit_address", std::numeric_limits<std::size_t>::max());
}

void Configuration::initHeadersToPass(const ptree& tree) {
    auto headers = tree.equal_range("headers_to_pass");
    for (auto it = headers.first; it != headers.second; ++it) {
        headersToPass.push_back(it->second.get<std::string>("name"));
    }
}

void Configuration::load(const yplatform::ptree& tree, std::shared_ptr<ymod_tvm::tvm2_module> module) {
    srand(static_cast<unsigned int>(time(NULL)));
    std::set_new_handler(badAllocHandler);

    const auto logger = std::make_shared<ContextLogger>(makeLoggerWithRequestId(""));

    try {
        initContentTypeDetector(contentTypeDetector, tree, logger);

        aliasClassList = AliasClassList(tree.get("mime_aliases", ""));

        initFactExtractor(factexParserConfig, factexConfig, tree, logger);

        bigLettersTrimThreshold = tree.get<unsigned>("big_letters_trim_threshold", 0);

        divTagDepthLimit = tree.get<unsigned>("div_tag_depth_limit",
                                              std::numeric_limits<unsigned>::max());

        initVdirect(vdirect, tree);
        initRecognizer(recognizer, tree);
        initSanitizer(sanitizer, tree);

        pa::async_profiler::init(500000, 16, tree.get("profiler_log", ""));

        initMacs();
        initMailStorage(tree);
        initExtVideoSingleton(tree, logger);
        initExternalTransformersSelector(tree);
        initPhishing(tree);
        initDariaViewMaker(lengthLimitAddress, tree);

        webattachServer = tree.get<std::string>("webattach_server");
        inlineAttachesTTL = std::chrono::hours(tree.get<unsigned>("inline_attaches_ttl_hours"));
        inlineAttachesShieldTimeout = std::chrono::seconds(tree.get<unsigned>("inline_attaches_shield_timeout_sec"));

        messageTextConfig.maxPartLength = tree.get<std::size_t>("message_text.max_part_length");

        const auto guard = tvm_guard::init(tree.get_child("tvm_guard"), module);
        tvmGuard = std::make_shared<tvm_guard::Guard<ymod_tvm::tvm2_module>>(guard);
        tvmGuardLogger = std::make_shared<TvmGuardLogger>(makeTvmGuardLogger());

        initHeadersToPass(tree);

        using namespace mail_getter::attach_sid;
        const KeyContainer keyContainer = parseKeyContainer(tree.get_child("attach_sid_keys"));
        keys = initKeys(tree.get_child("attach_sid_keys"), keyContainer);
        coroutineStackSize = tree.get<std::size_t>("coroutine_stack_size", 1048576);
        plainTextSanitize = tree.get<bool>("plain_text_sanitize", false);

        sanitizerStrategy = yamail::data::reflection::from_string<sanitizer::Strategy>(tree.get("sanitizer_strategy", "real"));

        MBODY_LOG_INFO(logger, log::message="successfully loaded config");
    } catch (const std::exception& e) {
        MBODY_LOG_ERROR(logger, log::message="exception while loading config", log::exception=e);
        throw;
    }
}

}
