
#include <fstream>
#include <boost/lexical_cast.hpp>

#include <internal/pa_log.h>

#include <internal/hilight/hilight.h>
#include <internal/config.h>
#include <internal/uudecode.h>
#include <internal/transformer_attributes.h>
#include <internal/message_tree.h>
#include <internal/mail_quote.h>
#include <internal/content_transformers.h>

namespace msg_body {

ContentTransformers::ContentTransformers(const Configuration&, TransformerAttributes& attrs,
                                         VdirectPtr vdirect)
    : attrs_(attrs)
    , vdirect_(vdirect)
{}

uudecode::Chunks uuDecode(std::string& content) {
    using namespace uudecode;
    const PaLog paLog(__FUNCTION__);

    ChunkWrappers chunkWrappers;
    GetUuencodeXMLRepresentation(content, chunkWrappers);

    Chunks chunks;
    std::transform(chunkWrappers.begin(), chunkWrappers.end(), std::back_inserter(chunks),
        std::mem_fun_ref(&ChunkWrapper::chunk));
    paLog.write();
    return chunks;
}

void ContentTransformers::applyUudecodeTransformer(MessagePart& part) const {
    uudecode::Chunks chunks = uuDecode(part.content);
    result_.uudecodeChunks = chunks;
    result_.hid = part.hid;
}

//------------------------ cut transformer -------------------------------------

void cutSignature(std::string& content) {
    const PaLog paLog(__FUNCTION__);
    std::istringstream is(content);
    std::ostringstream ret;
    std::string line;
    while (!is.eof()) {
        getline(is, line);
        if (line == "-- " || line == "-- \r") {
            break;
        }
        ret << line << std::endl;
    }
    content =  ret.str();
    paLog.write();
}

void ContentTransformers::applyCutTransformer(MessagePart& part) const {
    if (attrs_.flags.cutSignature) {
        cutSignature(part.content);
    }
}

//------------------ highlight transformer -------------------------------------

void mailToHtml(std::string& content, const std::string& textType) {
    const PaLog paLog(__FUNCTION__);

    if(textType != "html") {
        content = Mail2HTML(content);
    }
    paLog.write();
}

void highlight(std::string& content, const std::string& textType, EmbedInfos& embedList,
               const std::string& uid, VdirectPtr vdirect)
{
    const PaLog paLog(__FUNCTION__);
    hiliter::Params params(uid, vdirect);
    const std::string highlightSrc = (textType == "html" ? content : mailQuote(content));
    content = hiliter::hilite(highlightSrc, embedList, params);
    paLog.write();
}

void clearOriginalUrl(EmbedInfo& link) {
    link.original_url.clear();
}

void ContentTransformers::applyHighlightTransformer(MessagePart& part) const {
    if (attrs_.flags.outputAsCDATA || attrs_.flags.draft) {
        return;
    }

    if (attrs_.flags.ignoreHighlight) {
        mailToHtml(part.content, part.textType);
        return;
    }

    EmbedInfos links;
    highlight(part.content, part.textType, links, attrs_.uid, vdirect_);

    std::for_each(links.begin(), links.end(), &clearOriginalUrl);
    part.videoLinks = links;
    result_.videoLinks = links;
}

//----------------------------- main apply -------------------------------------

ContentTransformersResult ContentTransformers::apply(MessagePart& part) const {
    applyUudecodeTransformer(part);
    applyCutTransformer(part);
    applyHighlightTransformer(part);
    result_.isPhishing = part.isPhishing;
    return result_;
}

}
