#include <boost/utility.hpp>
#include <boost/regex.hpp>
#include <boost/range/algorithm/for_each.hpp>
#include <boost/range/algorithm/find_if.hpp>

#ifdef __clang__
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wsign-conversion"
#endif

#include <butil/StrUtils/StrUtils.h>
#include <butil/StrUtils/Iconv.h>
#include <butil/network/rfc2822.h>
#include <butil/network/idn.h>
#include <butil/datetime/date_utils.h>
#include <butil/network/utils.h>

#include <mimeparser/rfc2822date.h>
#include <mimeparser/rfc2047.h>
#include <mimeparser/ccnv.h>

#ifdef __clang__
#pragma clang diagnostic pop
#endif

#include <mail_getter/MessageAccess.h>
#include <mail_getter/UTFizer.h>
#include <internal/pa_log.h>

#include <internal/dkim_parser.h>
#include <internal/transformer_attributes.h>
#include <internal/message_walker.h>
#include <internal/daria_view_maker.h>

#include <macs/types.h>
#include <internal/medal_maker.h>

namespace msg_body {

DariaViewMaker::DariaViewMaker(const Configuration& config,
        TransformerAttributes& transformerAttributes,
        LogPtr logger,
        MessageAccess& messageAccess,
        IContentTypeDetector& contentTypeDetector,
        const AliasClassList& aliasClassList,
        unsigned trimThreshold,
        const std::string& timeZone,
        VdirectPtr vdirect,
        const Sanitizer& sanitizer,
        AsyncMacsServicePtr asyncMacsService,
        const InlineSpoofer& inlineSpoofer,
        const Recognizer::Wrapper& recognizer,
        const TextAsyncFactExtractor& asyncFactExtractor,
        macs::Attachments attachments)
    : transformerAttributes_(transformerAttributes)
    , logger_(logger)
    , config_(config)
    , messageAccess_(messageAccess)
    , transformers_(config,
        transformerAttributes_,
        messageContext_,
        logger,
        contentTypeDetector,
        aliasClassList,
        trimThreshold,
        vdirect,
        sanitizer,
        inlineSpoofer,
        recognizer,
        asyncFactExtractor)
    , messageTreeCreator_(messageAccess, contentTypeDetector, std::move(attachments))
    , timeZone_(timeZone)
    , asyncMacsService_(asyncMacsService)
    , recognizer(recognizer) {
}


void DariaViewMaker::clear() {
    textParts_.clear();
    rawParts_.clear();
    signatureParts_.clear();
    calendarParts_.clear();
    passbookParts_.clear();
    messageContext_.clear();
}

void setIsAttach(MessagePart& part) {
    part.isAttach = (part.headerStruct.contentDisposition() == "attachment");
}

void markMainTextPart(MessageParts& parts) {
    MessageParts::iterator part = boost::find_if(parts, !boost::bind(&MessagePart::isAttach, _1));
    if (part != parts.end()) {
        part->isMain = true;
    }
}

bool isTextBlock(const MessagePart& part) {
    return part.metaType == "text" && !part.isAttach;
}

MessageParts::const_iterator findLastTextPart(MessageParts& parts) {
    MessageParts::const_reverse_iterator revIt = std::find_if(parts.rbegin(), parts.rend(), &isTextBlock);
    return revIt == parts.rend()
        ? parts.begin()
        : (++revIt).base();
}

template <typename Iter1, typename Iter2>
bool isBefore(Iter1 first, Iter2 second) {
    return std::distance(first, second) > 0;
}

MessagePart makeBinaryPart(const MessagePart& part) {
    MessagePart res = part;
    res.metaType = "binary";
    return res;
}

void DariaViewMaker::recognizeParts(MessageParts& parts) {
    boost::for_each(parts, &setIsAttach);
    MessageParts::const_iterator lastTextPart = findLastTextPart(parts);

    const auto isCal = [](const MessagePart& part){ return part.metaType == "calendar"; };
    const auto isReal = [](const MessagePart& part){ return part.isMixedAttach; };
    const auto isRealCalendar = [isCal, isReal](const MessagePart& part){ return isCal(part) && isReal(part); };
    const bool hasRealCalendar = std::find_if(parts.begin(), parts.end(), isRealCalendar) != parts.end();

    for (MessageParts::const_iterator part = parts.begin(); part != parts.end(); ++part) {
        if (part->metaType == "signed") {
            signatureParts_.push_back(*part);
        } else if (part->metaType == "calendar") {
            if (!hasRealCalendar || isReal(*part)) {
                calendarParts_.push_back(*part);
                rawParts_.push_back(makeBinaryPart(*part));
            }
        } else if (part->metaType == "pkpass") {
            passbookParts_.push_back(*part);
            rawParts_.push_back(makeBinaryPart(*part));
        } else if (isTextBlock(*part)) {
            textParts_.push_back(*part);
        } else {
            rawParts_.push_back(*part);
            if (isBefore(part, lastTextPart)) {
                MessagePart inlineAttach = *part;
                inlineAttach.isAttach = true;
                textParts_.push_back(inlineAttach);
            }
        }
    }

    markMainTextPart(textParts_);
}

DariaBodyResult DariaViewMaker::formTextBody(MessagePart& part) {
    DariaBodyResult res;
    res.isAttach = part.isAttach;
    res.hid = part.hid;
    if (!part.isAttach) {
        part.content = messageAccess_.getBody(part.hid);
        res.transformerResult = transformers_.apply(part, part.metaType);
    }
    return res;
}

DariaBodiesResult DariaViewMaker::formTextBodies() {
    const PaLog paLog(__FUNCTION__);
    DariaBodiesResult res;
    std::transform(textParts_.begin(), textParts_.end(), std::back_inserter(res),
        boost::bind(&DariaViewMaker::formTextBody, *this, _1));
    paLog.write();
    return res;
}

std::string getAttachmentTransformType(const MessagePart& part) {
    return (part.metaType == "text") ? "binary" : part.metaType;
}

TransformersResultPtr DariaViewMaker::formAttachment(MessagePart& part) {
    const std::string transType = getAttachmentTransformType(part);
    if (transType == "narod" || transType == "pkpass") {
        part.content = messageAccess_.getBody(part.hid);
    }
    if (transType == "binary") {
        part.messageHeader = messageAccess_.getMessageHeaderParsed(rootHid);
    }
    return transformers_.apply(part, transType);
}

MessagePart makePart(macs::MimePart && mimePart) {
    MessagePart part;
    part.hid = mimePart.hid();
    part.contentType = MimeType(mimePart.contentType(), mimePart.contentSubtype());
    part.headerStruct = std::move(mimePart);
    part.metaType = classifyMetatype(part);
    return part;
}

MessageParts getWindatParts(const AsyncMacsServicePtr& asyncMacsService, const std::string& mid) {
    MessageParts windatAttaches;

    mail_errors::error_code ec;
    auto windatMimes = asyncMacsService->getWindatMimes(macs::Mids({mid}), ec);
    if (ec == sharpei::client::Errors::UidNotFound || windatMimes.begin() == windatMimes.end()) {
        return windatAttaches;
    } else if (ec) {
        throw std::runtime_error("Can't get windat parts by mid: " + ec.what());
    }
    std::transform(windatMimes.begin(), windatMimes.end(), std::back_inserter(windatAttaches), [&mid](auto& midStidWithMimes) {
        auto& mimes = std::get<2>(midStidWithMimes);
        if (mimes.empty()) {
            throw std::logic_error("getWindatMimes returned no mimes for mid=" + mid + " windat_st_id=" + std::get<1>(midStidWithMimes));
        }
        return makePart(std::move(mimes.front()));
    });
    return windatAttaches;
}

TransformPartsResult DariaViewMaker::tryFormWindatAttaches(MessagePart& part) {
    try {
        MessageParts parts = getWindatParts(asyncMacsService_, transformerAttributes_.mid);
        const auto attaches = parts | boost::adaptors::transformed([this](auto& arg) {
            return this->formAttachment(arg);
        });
        if (!attaches.empty()) {
            return {attaches.begin(), attaches.end()};
        }
    } catch (const MessagePartException& e) {
        MBODY_LOG_WARN(logger_, log::where_name="formDariaMessage", log::message="error in windat attach", log::exception=e, log::hid=e.partHid());
    } catch (const std::exception& e) {
        MBODY_LOG_WARN(logger_, log::where_name="formDariaMessage", log::message="error in windat attach", log::exception=e);
    }
    return {formAttachment(part)};
}

TransformPartsResult DariaViewMaker::formAttachments() {
    const PaLog paLog(__FUNCTION__);
    TransformPartsResult res;

    for (MessagePart& part: rawParts_) {
        if (isWindat(part) && asyncMacsService_) {
            const TransformPartsResult windatAttaches = tryFormWindatAttaches(part);
            res.insert(res.end(), windatAttaches.begin(), windatAttaches.end());
        } else {
            res.push_back(formAttachment(part));
        }
    }
    paLog.write();
    return res;
}

TransformersResultPtr DariaViewMaker::formSignature(MessagePart& part) {
    return transformers_.apply(part, part.metaType);
}

TransformPartsResult DariaViewMaker::formSignatures() {
    const PaLog paLog(__FUNCTION__);
    TransformPartsResult res;
    std::transform(signatureParts_.end(), signatureParts_.end(), std::back_inserter(res),
        boost::bind(&DariaViewMaker::formSignature, *this, _1));
    paLog.write();
    return res;
}

TransformersResultPtr DariaViewMaker::formCalendar(MessagePart& part) {
    part.content = messageAccess_.getBody(part.hid);
    return transformers_.apply(part, part.metaType);
}

TransformPartsResult DariaViewMaker::formCalendars() {
    const PaLog paLog(__FUNCTION__);
    TransformPartsResult res;
    std::transform(calendarParts_.begin(), calendarParts_.end(), std::back_inserter(res),
        boost::bind(&DariaViewMaker::formCalendar, *this, _1));
    paLog.write();
    return res;
}

TransformersResultPtr DariaViewMaker::formPassbookPackage(MessagePart& part) {
    part.content = messageAccess_.getBody(part.hid);
    return transformers_.apply(part, part.metaType);
}

TransformPartsResult DariaViewMaker::formPassbookPackages() {
    const PaLog paLog(__FUNCTION__);
    TransformPartsResult res;
    std::transform(passbookParts_.begin(), passbookParts_.end(), std::back_inserter(res),
        boost::bind(&DariaViewMaker::formPassbookPackage, *this, _1));
    paLog.write();
    return res;
}

void recodeAttr(MetaAttributes::value_type& attr, const Recognizer::Wrapper& recognizer) {
    const int result = UTFizer(recognizer).utfize(std::string(), attr.second);
    if (UTFizer::failed(result)) {
        std::ostringstream error;
        error << "Could not utfize string: \"" << attr.second << "\"";
        throw UTFizerException(error.str());
    }
}

MetaAttributes getHeaderMeta(MessageAccess& messageAccess, const std::string& hid,
                             const Recognizer::Wrapper& recognizer) {
    auto attrs = messageAccess.getMessageHeaderParsed(hid);
    boost::for_each(attrs, [&](auto& attr) {
        recodeAttr(attr, recognizer);
    });
    return attrs;
}

bool isSystemLetter(const MetaAttributes& headerMeta) {
    return headerMeta.find("to") == headerMeta.end();
}

DariaAddressResult formAddress(const rfc2822ns::address_iterator& addr, const std::string& direction,
                               const Recognizer::Wrapper& recognizer) {
    const std::string email = rfc2822ns::join_address(idna::decode(addr.local()), idna::decode(addr.domain()));
    std::string name = addr.display();
    if (!name.empty()) {
        std::string charset;
        name = mulca_mime::decode_rfc2047(name, charset);
        utfizeString(recognizer, name, charset);
        name = TStrUtils::removeSurroundQuotes(TStrUtils::unescape(name));
    } else {
        name = email;
    }
    DariaAddressResult res;
    res.direction = direction;
    res.name = name;
    res.email = email;
    return res;
}

DariaAddressesResult& operator+=(DariaAddressesResult& res, const DariaAddressResult& other) {
    res.push_back(other);
    return res;
}

DariaAddressesResult& operator+=(DariaAddressesResult& res, const DariaAddressesResult& other) {
    res.insert(res.end(), other.begin(), other.end());
    return res;
}

DariaAddressesResult DariaViewMaker::formAddresses(const std::string& addressStr, const std::string& direction) const {
    DariaAddressesResult res;
    std::string goodStr = TStrUtils::removeBadSymbols(mulca_mime::decode_numbered_entities(addressStr));
    try {
        for (rfc2822ns::address_iterator iter(goodStr), end; iter != end; ++iter) {
            const std::string addressDomain = iter.domain();
            const std::string addressLocal = iter.local();
            if(addressLocal.size() > config_.lengthLimitAddress) {
                MBODY_LOG_WARN(logger_, log::where_name=__FUNCTION__, log::message="address_local=" + addressLocal + ", direction=" 
                    + direction +  ", maximum allowed size exceeded");
            } else if(addressDomain.size() > config_.lengthLimitAddress) {
                MBODY_LOG_WARN(logger_, log::where_name=__FUNCTION__, log::message="address_domain=" + addressDomain + ", direction=" 
                    + direction + ", maximum allowed size exceeded");
            } else {
                res += formAddress(iter, direction, recognizer);
            }
        }
    } catch (const rfc2822ns::invalid_address& e) {
        MBODY_LOG_INFO(logger_, log::where_name=__FUNCTION__, log::message="direction=" + direction, log::exception=e);
    }
    return res;
}

DariaAddressesResult DariaViewMaker::formAddresses(const MetaAttributes& headerMeta, const std::string& attr) const {
    DariaAddressesResult res;
    MetaAttributes::const_iterator iter = headerMeta.find(attr);
    if (iter != headerMeta.end()) {
        res += formAddresses(iter->second, attr);
    }
    return res;
}

std::string extractNoReplyMessageId(const std::string& messageIdHeader) {
    static const boost::regex re("^<(\\d+).(remind|remind_noanswer)@");
    boost::smatch match;
    if (boost::regex_search(messageIdHeader, match, re)) {
        return match[1].str();
    } else {
        return "";
    }
}

DariaNoReplyResult formNoReplyNotification(MetaAttributes& headerMeta) {
    DariaNoReplyResult res;
    res.notification = extractNoReplyMessageId(headerMeta["message-id"]);
    return res;
}

DariaDateResult formDate(MetaAttributes& attrs, const std::string& timeZone) {
    DariaDateResult res;

    rfc2822::rfc2822date dt(attrs["date"]);
    const time_t unixtime = dt.unixtime();
    res.timestamp = static_cast<std::size_t>(unixtime*1000);

    const time_t unixtimeUserOffsetted = DateUtils::tztime(unixtime, timeZone);
    const time_t unixtimeUser = DateUtils::tztimeUtc(unixtimeUserOffsetted, "Europe/Moscow");
    res.userTimestamp = static_cast<std::size_t>(unixtimeUser*1000);

    return res;
}

void parseSubjectHeader(const std::string &src, std::string &subject,
                        const Recognizer::Wrapper& recognizer) {
    std::string charset;
    subject = mulca_mime::decode_rfc2047(src, charset);
    utfizeString(recognizer, subject, charset);
}

void parseDateHeader(const std::string &src, time_t &msgDate, long &timeZoneOffset) {
    rfc2822::rfc2822date dt(src);
    msgDate = dt.unixtime();
    timeZoneOffset = dt.offset();
}

void DariaViewMaker::parseAddressHeader(const std::string &src, std::string &email) const {
    DariaAddressesResult addresses = formAddresses(src, "from");
    if (addresses.size() > 0) {
        email = addresses[0].email;
    }
}

void DariaViewMaker::parseTransformerAttributesFromHeaders(TransformerAttributes& attrs, MessageAccess& messageAccess) const {
    const PaLog paLog(__FUNCTION__);
    auto headerMeta = getHeaderMeta(messageAccess, rootHid, recognizer);

    const std::size_t headersSize = std::accumulate(headerMeta.begin(), headerMeta.end(), 0, [] (std::size_t sum, auto& header) {
        return sum + header.first.size() + header.second.size();
    });
    MBODY_LOG_DEBUG(logger_, log::message="headers_size=" + std::to_string(headersSize));

    parseSubjectHeader(headerMeta["subject"], attrs.messageSubject, recognizer);
    parseDateHeader(headerMeta["date"], attrs.messageDate, attrs.timeZoneOffset);
    parseAddressHeader(headerMeta["from"], attrs.from);
    paLog.write();
}

Dkim getDkimCheckResult(MetaAttributes& attrs) {
    const PaLog paLog(__FUNCTION__);
    const char authHeader[] = "authentication-results";
    const std::string& header(attrs[authHeader]);
    const auto result = DkimParser().parse(header);
    paLog.write();
    return result;
}

DariaInfoResult DariaViewMaker::formInfo(const std::string& startHid) const {
    const PaLog paLog(__FUNCTION__);
    DariaInfoResult res;

    const std::string hid = startHid.empty() ? rootHid : startHid;
    auto headerMeta = getHeaderMeta(messageAccess_, hid, recognizer);

    res.stid = messageAccess_.getStId();
    res.references = headerMeta["references"];
    res.inReplyTo = headerMeta["in-reply-to"];
    res.messageId = headerMeta["message-id"];
    res.filterId = headerMeta["x-yandex-filter"];
    res.personalSpam = headerMeta["x-yandex-personal-spam"];
    res.spam = headerMeta["x-yandex-spam"];
    res.deliveredTo = headerMeta["delivered-to"];
    res.listUnsubscribe = headerMeta["list-unsubscribe"];
    res.liveMail = headerMeta["x-yandex-livemail"];

    if (!headerMeta["x-yandex-flight-direction"].empty()) {
        res.flightDirection = headerMeta["x-yandex-flight-direction"];
    }

    res.dateResult = formDate(headerMeta, timeZone_);

    if (isSystemLetter(headerMeta)) {
        res.addressesResult += formAddresses(transformerAttributes_.to, "to");
    } else {
        res.addressesResult += formAddresses(headerMeta, "to");
    }
    res.addressesResult += formAddresses(headerMeta, "from");
    res.addressesResult += formAddresses(headerMeta, "cc");
    res.addressesResult += formAddresses(headerMeta, "bcc");
    res.addressesResult += formAddresses(headerMeta, "reply-to");

    res.noReplyResult = formNoReplyNotification(headerMeta);
    res.dkim = getDkimCheckResult(headerMeta);

    MedalMaker medalMaker(asyncMacsService_, headerMeta, transformerAttributes_.mid);
    medalMaker.updateResult(res);

    paLog.write();
    return res;
}

bool isMessageBlock(const MessagePart& part) {
    return part.metaType == "message" ||
            (part.contentType.type() == "text" && part.contentType.subtype() == "rfc822-headers");
}

std::string extractOriginalMessageId(MessageParts& parts) {
    std::string messageId;
    MessageParts::const_iterator partIt = std::find_if(parts.begin(), parts.end(), &isMessageBlock);
    if (partIt != parts.end()) {
        const MetaAttributes& mta = partIt->messageHeader;
        MetaAttributes::const_iterator attrIter = mta.find("message-id");
        if (attrIter != mta.end()) {
            messageId = attrIter->second;
        }
    }
    return messageId;
}

CidParts getCidPartsInfo(const MessageParts& parts, const std::string &mid) {
    CidParts cidParts;
    for (MessageParts::const_iterator part = parts.begin(); part != parts.end(); ++part) {
        const auto& hs = part->headerStruct;
        const std::string& cid = hs.cid();
        if (!cid.empty()) {
            CidPartInfo cidInfo;
            cidInfo.stid = part->stid;
            cidInfo.hid = part->hid;
            cidInfo.mid = mid;

            cidInfo.name = !hs.fileName().empty() ? hs.fileName() : hs.name();

            cidParts[cid] = cidInfo;
        }
    }
    return cidParts;
}

bool checkIsOurMessage(MessageAccess& messageAccess, const Recognizer::Wrapper& recognizer) {
    auto headers = getHeaderMeta(messageAccess, rootHid, recognizer);
    const MetaAttributes::const_iterator header = headers.find("x-yandex-local");
    return header != headers.end() && header->second == "yes";
}

boost::optional<const MessagePart&> DariaViewMaker::findMainTextPart() const {
    const auto mainPart = std::find_if(textParts_.begin(), textParts_.end(),
        [] (const MessagePart& part) { return part.isMain; });
    if (mainPart == textParts_.end()) {
        return boost::none;
    } else {
        return *mainPart;
    }
}

DariaResult DariaViewMaker::formDariaMessage(MessageParts& messageParts, const std::string& hid) {
    DariaResult res;
    parseTransformerAttributesFromHeaders(transformerAttributes_, messageAccess_);
    messageContext_.originalMessageId_ = extractOriginalMessageId(messageParts);
    messageContext_.isOurMessage = checkIsOurMessage(messageAccess_, recognizer);
    recognizeParts(messageParts);
    messageContext_.cidParts = getCidPartsInfo(rawParts_, transformerAttributes_.mid);
    res.info = formInfo(hid);
    messageContext_.listUnsubscribe_ = !res.info.listUnsubscribe.empty();
    res.bodies = formTextBodies();
    res.attachments = formAttachments();
    res.signatures = formSignatures();
    res.calendars = formCalendars();
    res.passbooks = formPassbookPackages();
    return res;
}

DariaResult DariaViewMaker::formFallbackDariaMessage(MessageTree& messageTree,
        const std::string& hid, const std::string& excludeHid) {
    MessageWalker messageWalker;
    if (!messageWalker.make(messageTree, excludeHid)) {
        throw std::runtime_error("no valid parts found");
    }
    clear();
    MessageParts parts = messageWalker.parts();
    return formDariaMessageWithFallback(messageTree, parts, hid);
}

DariaResult DariaViewMaker::formDariaMessageWithFallback(MessageTree& messageTree,
        MessageParts& messageParts, const std::string& hid) {
    DariaResult res;
    try {
        res = formDariaMessage(messageParts, hid);
    } catch (const UTFizerException& e) {
        MBODY_LOG_WARN(logger_, log::where_name="formDariaMessage", log::message="could not utfize message", log::mid=transformerAttributes_.mid,
            log::uid=transformerAttributes_.uid, log::hid=hid);
        MBODY_LOG_DEBUG(logger_, log::where_name="formDariaMessage", log::message="could not utfize message", log::mid=transformerAttributes_.mid,
            log::uid=transformerAttributes_.uid, log::hid=hid, log::exception=e);
        throw std::runtime_error("could not utfize message");
    } catch (const MessagePartException& e) {
        MBODY_LOG_WARN(logger_, log::where_name="formDariaMessage", log::message="fallback due to error", log::exception=e,
            log::hid=e.partHid());
        return formFallbackDariaMessage(messageTree, hid, e.partHid());
    }
    const auto mainTextPart = findMainTextPart();
    if (!mainTextPart || mainTextPart->isComplete) {
        return res;
    }
    try {
        MBODY_LOG_WARN(logger_, log::where_name="formDariaMessage", log::message="fallback due to incomplete main part",
            log::hid=mainTextPart->hid);
        return formFallbackDariaMessage(messageTree, hid, mainTextPart->hid);
    } catch (const std::exception& e) {
        MBODY_LOG_WARN(logger_, log::where_name="formDariaMessage", log::message="exception in fallback",
            log::exception=e);
        return res;
    }
}

DariaResult DariaViewMaker::createDariaMessage(const std::string& hid) {
    const PaLog paLog(__FUNCTION__);
    MessageTree messageTree = messageTreeCreator_.create(hid);
    MessageWalker messageWalker;
    messageWalker.make(messageTree);
    MessageParts parts = messageWalker.parts();
    const auto result = formDariaMessageWithFallback(messageTree, parts, hid);
    paLog.write();
    return result;
}

}
