#include <boost/lexical_cast.hpp>
#include <boost/unordered_map.hpp>
#include <boost/algorithm/string.hpp>

#include <internal/pa_log.h>
#include <butil/butil.h>
#include <butil/xml/routines.h>
#include <butil/xml/entities.h>
#include <butil/xml/expat_transformer.h>

#include <internal/event_xml.h>

namespace msg_body {

const std::string fakeRootTag = "wrapper-to-remove";
const std::string inlineImageTag = "inline-image";
const std::string redirectorPrefix = "http://mail.yandex.ru/r?url=";

typedef boost::unordered_map<std::string, std::string> StringMap;

StringMap makeStringMap(const XML_Char** strs) {
    StringMap res;
    for (; *strs; strs += 2) {
        res[*strs] = *(strs+1);
    }
    return res;
}

class XmlHandler;

typedef ExpatTransformer<std::string::const_iterator, XmlHandler> EventXmlTransformer;

class XmlHandler {
public:
    XmlHandler(std::string& resContent, Cids& resCids,
               const std::string& source, bool isPhishing, const VideoParams& videoParams)
        : resContent_(resContent)
        , resCids_(resCids)
        , source_(source)
        , isPhishing_(isPhishing)
        , videoParams_(videoParams)
        , expat_(NULL)
        , isInsideA_(false)
        , isInsideWmiLink_(false)
        , isInsideWmiMailto_(false)
    {}

    static void startElement(void* userData, const XML_Char* name, const XML_Char** attrs) {
        handler(userData).onStartElement(name, makeStringMap(attrs));
    }

    static void endElement(void* userData, const XML_Char* name) {
        handler(userData).onEndElement(name);
    }

    static void defaultHandler(void* userData, const XML_Char*, int) {
        handler(userData).onText();
    }

    void setExpat(EventXmlTransformer* expat) {
        expat_ = expat;
    }

private:
    static XmlHandler& handler(void* userData) {
        EventXmlTransformer* expat = static_cast<EventXmlTransformer*>(userData);
        return expat->handler();
    }

    void onStartElement(const std::string& tag, const StringMap& attrs) {
        if (tag == fakeRootTag) {
            return;
        } else if (tag == "img") {
            onStartImg(attrs);
        } else if (tag == "a") {
            onStartA(attrs);
        } else if (tag == "span") {
            onStartSpan(attrs);
        } else {
            appendCurrentEvent();
        }
    }

    void onStartImg(const StringMap& attrs) {
        StringMap newAttrs = attrs;
        StringMap::iterator srcIter = newAttrs.find("src");
        if (srcIter == newAttrs.end() || srcIter->second.substr(0, 4) != "cid:") {
            appendCurrentEvent();
            return;
        }
        const std::string cid = srcIter->second.substr(4);
        newAttrs.erase(srcIter);
        resContent_.append(xmlWrap(inlineImageTag, cid, newAttrs));
        resCids_.insert(cid);
    }

    void onStartA(const StringMap& attrs) {
        isInsideA_ = true;
        if (isPhishing_) {
            appendRedirectedLink(attrs);
        } else {
            appendCurrentEvent();
        }
    }

    void onStartSpan(const StringMap& attrs) {
        StringMap::const_iterator classIter = attrs.find("class");
        if (classIter == attrs.end()) {
            appendCurrentEvent();
            return;
        }
        if (classIter->second == "wmi-link") {
            onStartSpanWmiLink(attrs);
        } else if (classIter->second == "wmi-mailto") {
            onStartSpanWmiMailto();
        } else if (classIter->second == "wmi-video-link") {
            onStartSpanWmiVideoLink(attrs);
        } else {
            appendCurrentEvent();
        }
    }

    void onStartSpanWmiLink(const StringMap& attrs) {
        isInsideWmiLink_ = true;
        StringMap::const_iterator showIter = attrs.find("show");
        wmiLinkText_ = (showIter == attrs.end() ? "" : showIter->second);
    }

    void onStartSpanWmiMailto() {
        isInsideWmiMailto_ = true;
    }

    void onStartSpanWmiVideoLink(const StringMap& _attrs) {
        StringMap attrs = _attrs;
        StringMap::iterator titleIter = attrs.find("title");
        if (titleIter == attrs.end()) {
            appendCurrentEvent();
            return;
        }

        int videoId = 0;
        try {
            videoId = boost::lexical_cast<int>(titleIter->second);
        } catch(const boost::bad_lexical_cast&) {
            throw std::runtime_error("Can't cast video_link id: " + titleIter->second);
        }

        VideoParams::const_iterator videoIter = videoParams_.find(videoId);
        if (videoIter == videoParams_.end()) {
            appendCurrentEvent();
            return;
        }
        attrs.erase(titleIter);
        attrs["params"] = videoIter->second;
        resContent_.append(xmlOpenTag("span", attrs));
    }

    void onEndElement(const std::string& tag) {
        if (tag == fakeRootTag) {
            return;
        }
        if (tag == "a") {
            isInsideA_ = false;
        }
        if (isInsideWmiLink_) {
            onEndSpanWmiLink();
        } else if (isInsideWmiMailto_) {
            onEndSpanWmiMailto();
        } else {
            appendCurrentEvent();
        }
    }

    void onEndSpanWmiLink() {
        isInsideWmiLink_ = false;
        if (wmiLinkHref_.empty()) {
            return;
        }
        decodeXmlEntities(wmiLinkHref_);
        std::string text = (wmiLinkText_.empty() ? wmiLinkHref_ : wmiLinkText_);
        fixBadLink(wmiLinkHref_);
        redirectLink(wmiLinkHref_);
        StringMap attrs;
        attrs["href"] = wmiLinkHref_;
        resContent_.append(xmlWrap("a", encodeXmlEntities(text), attrs));
        wmiLinkHref_.clear();
    }

    void onEndSpanWmiMailto() {
        isInsideWmiMailto_ = false;
    }

    void onText() {
        if (isInsideA_ && (isInsideWmiLink_ || isInsideWmiMailto_)) {
            appendCurrentEvent();
            return;
        }
        if (isInsideWmiLink_) {
            onTextInsideWmiLink();
            return;
        }
        if (isInsideWmiMailto_) {
            onTextInsideWmiMailto();
            return;
        }
        appendCurrentEvent();
    }

    void onTextInsideWmiLink() {
        wmiLinkHref_.append(getCurrentEvent());
    }

    void onTextInsideWmiMailto() {
        const std::string text = getCurrentEvent();
        StringMap attrs;
        attrs["href"] = "mailto:" + text;
        resContent_.append(xmlWrap("a", text, attrs));
    }

    void appendCurrentEvent() {
        resContent_.append(getCurrentEvent());
    }

    void appendRedirectedLink(const StringMap& _attrs) {
        StringMap attrs = _attrs;
        StringMap::iterator hrefIter = attrs.find("href");
        if (hrefIter == attrs.end() || hrefIter->second.substr(0, 7) == "mailto:") {
            appendCurrentEvent();
            return;
        }
        redirectLink(hrefIter->second);
        resContent_.append(xmlOpenTag("a", attrs));
    }

    std::string getCurrentEvent() {
        if (!expat_) {
            throw std::runtime_error("EventXmlTransformer expat_ is not set");
        }
        auto offset = XML_GetCurrentByteIndex(expat_->parser());
        auto length = XML_GetCurrentByteCount(expat_->parser());
        std::string::const_iterator begin = source_.begin() + offset;
        std::string::const_iterator end = source_.begin() + offset + length;
        return std::string(begin, end);
    }

    void fixBadLink(std::string& url) {
        using namespace boost::algorithm;
        if (url.substr(0, 2) != "//"
                && to_lower_copy(url.substr(0, 4)) != "http"
                && to_lower_copy(url.substr(0, 3)) != "ftp") {
            url = "http://" + url;
        }
    }

    void redirectLink(std::string& url) {
        if (isPhishing_) {
            url = redirectorPrefix + encode_url(url);
        }
    }

    // output
    std::string& resContent_;
    Cids& resCids_;

    // input params
    const std::string source_;
    const bool isPhishing_;
    const VideoParams videoParams_;

    // state variables
    EventXmlTransformer* expat_;
    bool isInsideA_;
    bool isInsideWmiLink_;
    bool isInsideWmiMailto_;
    std::string wmiLinkText_;
    std::string wmiLinkHref_;
};

EventXmlResult eventXmlTransform(const std::string& content, bool isPhishing, const EmbedInfos& videoLinks) {
    const PaLog paLog(__FUNCTION__);
    EventXmlResult res;
    std::string wrappedContent = xmlWrap(fakeRootTag, content);
    XmlHandler xmlHandler(res.content_, res.cids_, wrappedContent, isPhishing, getVideoParams(videoLinks));
    EventXmlTransformer expat(wrappedContent.begin(), wrappedContent.end(), xmlHandler);
    xmlHandler.setExpat(&expat);
    paLog.write();
    if(!expat.transform()) {
        throw std::runtime_error("XML parsing failed: " + expat.error());
    }
    return res;
}

}
