#include <map>
#include <stdexcept>
#include <boost/lexical_cast.hpp>
#include <butil/datetime/date_utils.h>
#include <butil/xml/helpers.h>
#include <internal/ext_video.h>

namespace msg_body {

ExtVideo ExtVideoSingleton::_instance;

static bool save_attr(xmlNodePtr n, const char *name, std::string& var) {
    xmlAttrPtr attr = xmlHasProp(n, XmlHelpers::c2x(name));
    if(attr) {
        var = XmlHelpers::x2c(attr->children->content);
        return true;
    }
    return false;
}

static bool save_attr(xmlNodePtr n, const char *name, int& var) {
    xmlAttrPtr attr = xmlHasProp(n, XmlHelpers::c2x(name));
    if(attr) {
        var = boost::lexical_cast<int>(XmlHelpers::x2c(attr->children->content));
        return true;
    }
    return false;
}

static void loadRegexp(xmlNodePtr node, TRegexp& regexp) {
    std::string text;
    if(save_attr(node, "text", text)) {
        const char* err;
        int err_off;
        regexp.first = pcre_compile(text.c_str(), 0, &err, &err_off, NULL);
        if (!regexp.first)
            throw std::runtime_error("regexp compile failed " + text
                                     + " error " + std::string(err) + " at "
                                     + boost::lexical_cast<std::string>(err_off));
    }
    for(xmlNodePtr var = node->children; var != NULL; var = var->next) {
        int order;
        std::string name;
        if(save_attr(var, "order", order) && save_attr(var, "name", name)) {
            if(regexp.second.size() <= static_cast<size_t>(order))
                regexp.second.resize(static_cast<size_t>(order) + 1ul);
            regexp.second[static_cast<size_t>(order)] = name;
        }
    }
}

static void loadPattern(xmlNodePtr node, TParamString& pattern) {
    save_attr(node, "text", pattern.text);
    std::string name;
    for(xmlNode *var = node->children; var != NULL; var = var->next)
        if(save_attr(var, "name", name))
            pattern.params.push_back(name);
}

static void loadEmbedParams(xmlNodePtr node, TVideoHosting& hosting) {
    save_attr(node, "url", hosting.url);
    save_attr(node, "name", hosting.name);
    save_attr(node, "width", hosting.width);
    save_attr(node, "height", hosting.height);
    for(xmlNodePtr n = node->children; n != NULL; n = n->next) {
        if(n->type == XML_ELEMENT_NODE) {
            std::string name(XmlHelpers::x2c(n->name));
            if(name == "regexp") {
                TRegexp regexp;
                loadRegexp(n, regexp);
                hosting.regexpList.push_back(regexp);
            }
            if(name == "player_url")
                loadPattern(n, hosting.player_url);
            if(name == "flashvars")
                loadPattern(n, hosting.flashvars);
        }
    }
    if(hosting.player_url.text.empty() && hosting.regexpList.empty())
        throw std::runtime_error("player_url or regexp empty in hosting "
                                 + hosting.url);
}

int ExtVideo::loadHostings(const std::string& location) {
    xmlDocPtr ext_video = xmlReadFile(location.c_str(), NULL, 0);
    if (!ext_video)
        throw std::runtime_error("can't load ext_video config");

    xmlNodePtr root = xmlDocGetRootElement(ext_video);
    for(xmlNodePtr n = root->children; n != NULL; n = n->next) {
        std::string name(XmlHelpers::x2c(n->name));
        if((n->type == XML_ELEMENT_NODE) && (name == "hosting")) {
            TVideoHosting host;
            loadEmbedParams(n, host);
            hosting_list.push_back(host);
        }
    }
    xmlFreeDoc(ext_video);
    return true;
}

bool replace_vars(const std::list<std::string>& params, std::string& txt,
                  std::map<std::string, std::string>& vars) {
    typedef std::list<std::string>::const_iterator const_iterator;
    for(const_iterator it = params.begin(); it != params.end(); it++) {
        std::string pattern = "${" + *it + "}";
        std::string::size_type posT = txt.find ( pattern );
        std::string::size_type sizeT = pattern.size();
        if((posT == std::string::npos) || (vars.find((*it)) == vars.end()))
            return false;
        txt.replace(posT, sizeT, vars[*it]);
    }
    return true;
}

static bool createEmbed(const std::string& url, msg_body::EmbedInfo& embed, TVideoHosting& hosting) {
    embed.width = hosting.width;
    embed.height = hosting.height;
    embed.player_url = hosting.player_url.text;
    embed.flashvars = hosting.flashvars.text;
    embed.original_url = url;
    embed.hosting_name = hosting.name;

    std::map<std::string, std::string> vars;

    typedef TRegexpList::iterator iterator;
    for(iterator it = hosting.regexpList.begin(); it != hosting.regexpList.end(); it++) {
        int matchCount;
        int ovector[768];
        if((matchCount = pcre_exec(it->first, NULL, url.c_str(), static_cast<int>(url.size()), 0,
                                   0, ovector, 256)) >= 0) {
            for(size_t i = 0; i < it->second.size(); i++ ) {
                if(it->second[i].empty()) continue;
                int varPos = ovector[(i * 2) + 2];
                int varSize = ovector[(i * 2) + 3] - varPos;
                vars[it->second[i]] = url.substr(static_cast<std::string::size_type>(varPos),
                                                static_cast<std::string::size_type>(varSize));
            }
        }
    }
    return replace_vars(hosting.player_url.params, embed.player_url, vars)
        && replace_vars(hosting.flashvars.params, embed.flashvars, vars);
}

int ExtVideo::checkUrl(const std::string &url, msg_body::EmbedInfo &embed ) {
    typedef std::list<TVideoHosting>::iterator iterator;
    for(iterator it = hosting_list.begin(); it != hosting_list.end(); it++)
        if(url.find(it->url) != std::string::npos)
            if(createEmbed(url, embed, *it)) return true;
    return false;
}

ExtVideo::~ExtVideo() {
    typedef std::list<TVideoHosting>::iterator iterator;
    for(iterator it = hosting_list.begin(); it != hosting_list.end(); it++)
        it->freeRegexps();
}


void TVideoHosting::freeRegexps(void) {
    typedef TRegexpList::iterator iterator;
    for(iterator i = regexpList.begin(); i != regexpList.end(); ++i)
        pcre_free(i->first);
}

}
