#include <internal/extract_headers.h>
#include <mail_getter/UTFizer.h>
#include <mail_getter/MessageAccess.h>
#include <internal/server/request_context.h>
#include <internal/config.h>
#include <internal/macs/factory.h>
#include <internal/mulcagate/logging.h>
#include <internal/headers_parser.h>
#include <macs/types.h>

namespace msg_body {

struct MessageAccessParamsWithMid {
    mail_getter::MessageAccessParams msgAccessParams;
    macs::Mid mid;
};

MessagesHeadersParams parseMessagesHeadersParams(const RequestContext& req) {
    MessagesHeadersParams res;
    res.mids = req.getRequiredArgs("mid");
    res.uid = req.getRequiredNonEmptyArg("uid");
    res.requestId = req.requestId();
    return res;
}

static MacsParams getMacsParams(const MessagesHeadersParams& params) {
    MacsParams res;
    res.uid = params.uid;
    res.requestId = params.requestId;
    return res;
}


std::string midsToString(const macs::MidVec& mids) {
    return boost::algorithm::join(mids, " ");
}

static std::vector<MessageAccessParamsWithMid> getMessagesAccessParams(
        const AsyncMacsServicePtr& asynMacsService,
        const MessagesHeadersParams& params, LogPtr logger) {
    if (asynMacsService) {
        mail_errors::error_code ec;
        auto mimes = asynMacsService->getMimes(macs::Mids(params.mids.begin(), params.mids.end()), ec);
        auto logger_ =  getLoggerWithUidMids(*logger, params.uid, std::cref(params.mids));
        if (ec == sharpei::client::Errors::UidNotFound) {
            LOGDOG_(logger_, error, log::where_name="extract headers", log::message="cant't create message access with current uid");
            throw std::runtime_error(ec.what());
        } else if (ec) {
            LOGDOG_(logger_, error, log::where_name="extract headers", log::message="can't create message access by uid");
            throw std::runtime_error(ec.what());
        } else if (mimes.size() != params.mids.size()) {
            LOGDOG_(logger_, error, log::where_name="extract headers", log::message="getMimes returns records count not equal mids count: records_count=" + 
                std::to_string(mimes.size()) + ", mids_count=" + std::to_string(params.mids.size()));
            throw std::logic_error("returns records count not equal mids count");
        }
        std::vector<MessageAccessParamsWithMid> res;
        for (auto& mime : mimes) {
            auto mid = std::move(std::get<0>(mime));
            res.push_back({ mail_getter::getMessageAccessParams(std::move(mime)), std::move(mid) });
        }
        return res;
    }
    return std::vector<MessageAccessParamsWithMid>();
}

Mids2MessageHeaders extractMessagesHeaders(const Configuration& config, const MessagesHeadersParams& params,
        LogPtr logger, YieldCtx yc) {
    const PaLog paLog(__FUNCTION__, params.requestId);
    const auto mgLogger = MulcagateLog::create(params.requestId);
    const auto storageService = config.mailStorage->createService(mgLogger, tvm::Ticket(),
                                                                  params.requestId);

    const auto asyncMacsService = getAsyncMacsService(*config.maildb, getMacsParams(params), JournalParams{}, logger, yc);
    Mids2MessageHeaders messagesWithHeaders;
    auto msgsAccessesParamsWithMids = getMessagesAccessParams(asyncMacsService, params, logger);

    for (const auto& msgAccessParamsWithMid : msgsAccessesParamsWithMids) {
        auto msgAccess = storageService->createMessageAccess(msgAccessParamsWithMid.msgAccessParams,
            *config.recognizer, yc);
        MessageHeaders headers;
        auto hdr = msgAccess->getHeader(rootHid);
        HeaderParser hp(headers, hdr);
        hp.parse(0, 0, hdr.size());
        messagesWithHeaders.emplace(std::move(msgAccessParamsWithMid.mid), std::move(headers));
    }
    paLog.write();
    return messagesWithHeaders;
}

}
