#include <butil/StrUtils/Encoding.h>
#include <mail_getter/SimpleAttachment.h>

#include <internal/inline_spoofer.h>
#include <mail/message_body/include/internal/config.h>

namespace msg_body {

DraftInlineSpoofer::DraftInlineSpoofer(const Configuration& config,
        LogPtr logger,
        mail_getter::ServicePtr storageService,
        MessageAccess& messageAccess,
        const mail_getter::attach_sid::Keys& keys,
        YieldCtx yc)
    : config_(config)
    , logger_(logger)
    , storageService_(storageService)
    , messageAccess_(messageAccess)
    , keys_(keys)
    , yc_(yc)
{}

std::string DraftInlineSpoofer::getUrl(const CidPartInfo& cidInfo) const {
    using namespace mail_getter;
    std::string body;
    MetaPart hs;
    try {
        body = messageAccess_.getBody(cidInfo.hid);
        hs = messageAccess_.getHeaderStruct(cidInfo.hid);
    } catch (const std::exception& e) {
        MBODY_LOG_ERROR(logger_, log::where_name="inline_spoof", log::hid=cidInfo.hid,
            log::message="exception while downloading inline attach", log::exception=e);
        return "";
    }

    Encoding::decode(hs.encoding(), body);
    auto mimeType = getContentType(hs, cidInfo.hid, cidInfo.name, body).toString();
    SimpleAttachment attachment(cidInfo.name, std::move(mimeType), std::move(body));
    auto storage = storageService_->createAttachmentStorage(config_.webattachServer);

    std::string tmpStid;
    AttachmentStorage::MetaInfo metaInfo;
    int errorCode = storage->add(attachment, tmpStid, metaInfo, config_.inlineAttachesTTL,
            config_.inlineAttachesShieldTimeout, keys_, yc_);

    if (errorCode) {
        MBODY_LOG_ERROR(logger_, log::where_name="inline_spoof", log::hid=cidInfo.hid,
            log::message="error uploading new inline attach: status=" + std::to_string(errorCode));
        return "";
    }

    const std::string classUrl = "yandex_class=yandex_new_inline_" + metaInfo.sid;

    return metaInfo.viewLargeUrl + "&" + classUrl;
}

MimeType DraftInlineSpoofer::getContentType(MetaPart& hs, const mail_getter::Hid& hid,
        const std::string& filename, const std::string& body) const {
    try {
        if (hs.contentType().size() && hs.contentSubtype().size()) {
            return MimeType(hs.contentType(), hs.contentSubtype());
        }
    } catch (const std::exception& e) {
        MBODY_LOG_ERROR(logger_, log::where_name="inline_spoof", log::hid=hid,
            log::message="exception while accessing part content_type", log::exception=e);
    }

    return config_.contentTypeDetector->detect(filename, body);
}

std::string MainInlineSpoofer::getUrl(const CidPartInfo& cidInfo) const {
    const std::string classParam = "yandex_class=yandex_inline_content_" + cidInfo.stid + "_" +
        cidInfo.hid + "_" + cidInfo.mid;
    const std::string name = encode_url(cidInfo.name);
    return "../message_part/" + name + "?_uid=" + uid_ + "&amp;hid=" + cidInfo.hid +
        "&amp;ids=" + cidInfo.mid + "&amp;name=" + name + "&amp;" + classParam;
}

InlineSpooferPtr getInlineSpoofer(const Configuration& config, LogPtr logger,
        const TransformerAttributes& attrs, mail_getter::ServicePtr storageService,
        MessageAccess& messageAccess, YieldCtx yc) {
    if (attrs.flags.draft) {
        return std::make_unique<DraftInlineSpoofer>(config, logger, storageService, messageAccess, config.keys, yc);
    } else {
        return std::make_unique<MainInlineSpoofer>(attrs.uid);
    }
}

} // namespace

