#include <string>
#include <butil/xml/entities.h>
#include <internal/mail_quote/parser.h>
#include <internal/mail_quote/visitor.h>

namespace msg_body {

using std::string;

struct MailQuote {
    MailQuote(const string& s, string& r)
        : src(s), result(r), i(0), spaces(false)
    {}

    void copy_escape(void);
    void operator()(bool);

private:
    const string& src;
    string& result;
    string indent;
    string::size_type i;
    bool spaces;
    const XmlEntities entities;
};

void MailQuote::copy_escape(void)
{
    if(src[i] == ' ') {
        if(spaces) result += "&#160;";
        else result += " ";
        spaces = true;
    } else {
        spaces = false;
        if(src[i] == '<' || src[i] == '>' || src[i] == '&') {
            result += entities.printEntity(src[i]);
        } else if(src[i] == '\r') return;
        else if(src[i] == '\n') result += "<br />";
        else result += src[i];
    }
}

void MailQuote::operator()(bool)
{
    for(; i < src.size(); ++i) {
        for(;i < src.size() && src[i] != '\n'; ++i) copy_escape();
        result += "\n";
    }
}

static const std::string beginPgpSignedMessage = "-----BEGIN PGP SIGNED MESSAGE-----";
static const std::string beginPgpSignature = "-----BEGIN PGP SIGNATURE-----";
static const std::string endPgpSignature = "-----END PGP SIGNATURE-----";
static const std::string blockquoteOpen = "<blockquote class=\"wmi-pgp\">";
static const std::string blockquoteClose = "</blockquote>";

std::string::size_type findEmptyLine(const std::string& str, std::string::size_type start = 0) {
    std::string::size_type pos = str.find("\r\n\r\n", start);
    if (pos == std::string::npos) {
        pos = str.find("\n\n", start);
        if (pos == std::string::npos) {
            pos = str.find("<br/><br/>", start);
            if (pos == std::string::npos) {
                pos = str.find("<br /><br />", start);
            }
        }
    }
    return pos;
}

void removePGPHeader(std::string& str, std::string::size_type pos) {
    std::string::size_type emptyLinePos = findEmptyLine(str, pos);
    if (emptyLinePos == std::string::npos) {
        return;
    }
    str.erase(pos, emptyLinePos - pos);
}

size_t wrapPGPSignature(std::string& str, std::string::size_type pos) {
    std::string::size_type signaturePos = str.find(beginPgpSignature, pos);
    std::string::size_type endPos = str.find(endPgpSignature, signaturePos);
    if (signaturePos != std::string::npos && endPos != std::string::npos) {
        str.insert(signaturePos, blockquoteOpen);
        const std::size_t blockquoteClosePos = endPos + endPgpSignature.length() + blockquoteOpen.length();
        str.insert(blockquoteClosePos, blockquoteClose);
        return blockquoteClosePos + blockquoteClose.length();
    } else {
        return std::string::npos;
    }
}

void hidePGP(std::string& str) {
    std::string::size_type pos = str.find(beginPgpSignedMessage);
    while (pos != std::string::npos) {
        removePGPHeader(str, pos);
        pos = wrapPGPSignature(str, pos);
        pos = str.find(beginPgpSignedMessage, pos);
    }
}

std::string mailQuote(const std::string& text) {
    const auto tokens = mail_quote::tokenize(text);
    mail_quote::Parser parse(tokens.begin(), tokens.end());
    mail_quote::ParagraphVisitor paragraphVisitor(tokens.end());
    mail_quote::Visitor visitor(paragraphVisitor);
    parse(visitor);
    auto result = visitor.result();

    hidePGP(result);

    return result;
}

std::string Mail2HTML(const std::string& text) {
    std::string result;
    MailQuote m_q(text, result);
    m_q(false);
    return result;
}

} // namespace msg_body
