#include <internal/mail_quote/paragraph_visitor.h>

#include <sstream>

namespace msg_body {
namespace mail_quote {

const std::string ParagraphVisitor::breakLine = "<br />";
const std::string ParagraphVisitor::beginQuotation = "<blockquote class=\"wmi-quote\">";
const std::string ParagraphVisitor::endQuotation = "</blockquote>";
const std::string ParagraphVisitor::beginText = "<p>";
const std::string ParagraphVisitor::endText = "</p>";
const std::string ParagraphVisitor::space = "&#160;";
const std::string ParagraphVisitor::tab = "&#160;&#160;&#160;&#160;&#160;&#160;&#160;&#160;";
const std::string ParagraphVisitor::beginSignature = "<span class=\"wmi-sign\">";
const std::string ParagraphVisitor::endSignature = "</span>";
const std::string ParagraphVisitor::signatureSeparator = "-- <br />";

ParagraphVisitor::Paragraph& ParagraphVisitor::Paragraph::operator +=(std::string content) {
    this->content += std::move(content);
    return *this;
}

std::string ParagraphVisitor::Paragraph::dump() {
    std::stringstream stream;
    switch (type) {
        case Type::Text:
            stream << beginText << std::move(content) << endText;
            break;
        case Type::Quotation:
            stream << beginQuotation << std::move(content) << endQuotation;
            break;
        case Type::Signature:
            stream << beginSignature << std::move(content) << endSignature;
            break;
        default:
            stream << std::move(content);
            break;
    }
    type = Type::None;
    return stream.str();
}

void ParagraphVisitor::visit() {
    isLineContainsOnlySpaceSymbols = false;
    line.clear();
    paragraphsStack.clear();
    paragraphsSequence.clear();
    result_.clear();
}

void ParagraphVisitor::leave() {
    std::stringstream stream;
    Paragraph signature;
    for (auto&& paragraph : paragraphsSequence) {
        switch (paragraph.type) {
            case Paragraph::Type::Signature:
                if (signature.type == Paragraph::Type::None) {
                    signature = std::move(paragraph);
                } else {
                    signature += paragraph.content;
                }
                break;
            case Paragraph::Type::Quotation:
                if (signature.type == Paragraph::Type::None) {
                    stream << paragraph.dump();
                } else {
                    signature += paragraph.dump();
                }
                break;
            default:
                if (signature.type != Paragraph::Type::None) {
                    stream << signature.dump();
                }
                stream << paragraph.dump();
                break;
        }
    }
    if (signature.type != Paragraph::Type::None) {
        stream << signature.dump();
    }
    result_ = stream.str();
}

void ParagraphVisitor::visitParagraph() {
    paragraphsStack.push_back(Paragraph());
}

void ParagraphVisitor::leaveParagraph() {
    auto paragraph = std::move(paragraphsStack.back());
    paragraphsStack.pop_back();
    if (paragraphsStack.empty()) {
        paragraphsSequence.emplace_back(std::move(paragraph));
    } else {
        paragraphsStack.back() += paragraph.dump();
    }
}

void ParagraphVisitor::visitQuotation() {
    paragraphsStack.back().type = Paragraph::Type::Quotation;
}

void ParagraphVisitor::visitText() {
    paragraphsStack.back().type = Paragraph::Type::Text;
}

void ParagraphVisitor::visitSignature() {
    paragraphsStack.back().type = Paragraph::Type::Signature;
}

void ParagraphVisitor::visitLine() {
    isLineContainsOnlySpaceSymbols = false;
    line.clear();
}

void ParagraphVisitor::leaveLine() {
    if (!isLineContainsOnlySpaceSymbols && !line.empty()) {
        paragraphsStack.back() += line;
    } else {
        paragraphsStack.back() += breakLine;
    }
}

void ParagraphVisitor::visitSymbol(Iterator current) {
    if (current->isSpace()) {
        if (line.empty()) {
            isLineContainsOnlySpaceSymbols = true;
        }
    } else {
        isLineContainsOnlySpaceSymbols = false;
    }

    switch (current->type()) {
        case Token::Type::LineDelimiter:
            line += breakLine;
            break;
        case Token::Type::Ampersand:
        case Token::Type::LessThan:
        case Token::Type::GreaterThan:
            for (const auto symbol : current->value()) {
                line += entities.printEntity(symbol);
            }
            break;
        case Token::Type::CarriageReturn:
            break;
        case Token::Type::SignatureSeparator:
            line += signatureSeparator;
            break;
        default:
            if (isLineContainsOnlySpaceSymbols) {
                switch (current->type()) {
                    case Token::Type::Space:
                        line += space;
                        break;
                    case Token::Type::Tab:
                        line += tab;
                        break;
                    default:
                        line += current->value();
                }
            } else {
                line += current->value();
            }
    }
}

void ParagraphVisitor::leaveSymbol(const Range& range) {
    if (!range.empty() && range.end() == end
            && range.begin()->type() != Token::Type::LineDelimiter
            && range.begin()->type() != Token::Type::SignatureSeparator) {
        line += breakLine;
    }
}

} // namespace mail_quote
} // namespace msg_body
