#include <internal/mail_quote/parser.h>
#include <iostream>

namespace msg_body {
namespace mail_quote {

void Parser::operator ()(AbstractVisitor& visitor) {
    visitor.visit(quotationLevel, begin);
    for (current = begin; current != end; ++current) {
        state = handleToken(visitor);
    }
    handleEnd(visitor);
    visitor.leave(quotationLevel, range(begin, end));
}

Parser::State Parser::handleToken(AbstractVisitor& visitor) {
    switch (state) {
        case State::NewLine:
            return whenNewLine(visitor);
        case State::Quotation:
            return whenQuotation(visitor);
        case State::Text:
            return whenText(visitor);
        default:
            return state;
    }
}

void Parser::handleEnd(AbstractVisitor& visitor) {
    if (state != State::NewLine && textBegin != end) {
        if (quotationLevel == 0) {
            visitor.leaveText(quotationLevel, textRange());
            textBegin = end;
        } else {
            visitor.leaveQuotation(quotationLevel, textRange());
            textBegin = end;
            quotationLevel = 0;
        }
    } else {
        quotationLevel = 0;
    }
}

Parser::State Parser::whenNewLine(AbstractVisitor& visitor) {
    switch (current->type()) {
        case Token::Type::GreaterThan:
            quotationLevel = 1;
            return State::Quotation;
        case Token::Type::LineDelimiter:
            visitor.visitText(quotationLevel, current);
            visitLeaveSymbol(visitor);
            visitor.leaveText(quotationLevel, symbolRange());
            signature = false;
            return State::NewLine;
        case Token::Type::SignatureSeparator:
            signature = true;
            visitor.visitSignature(quotationLevel, current);
            visitLeaveSymbol(visitor);
            visitor.leaveSignature(quotationLevel, symbolRange());
            return State::NewLine;
        default:
            if (signature) {
                visitor.visitSignature(quotationLevel, current);
            } else {
                visitor.visitText(quotationLevel, current);
            }
            visitLeaveSymbol(visitor);
            textBegin = current;
            return State::Text;
    }
}

Parser::State Parser::whenQuotation(AbstractVisitor& visitor) {
    switch (current->type()) {
        case Token::Type::GreaterThan:
            ++quotationLevel;
            return state;
        case Token::Type::LineDelimiter:
            visitor.visitQuotation(quotationLevel, current);
            visitLeaveSymbol(visitor);
            visitor.leaveQuotation(quotationLevel, symbolRange());
            quotationLevel = 0;
            return State::NewLine;
        default:
            visitor.visitQuotation(quotationLevel, current);
            visitLeaveSymbol(visitor);
            textBegin = current;
            return State::Text;
    }
}

Parser::State Parser::whenText(AbstractVisitor& visitor) {
    switch (current->type()) {
        case Token::Type::LineDelimiter:
            visitLeaveSymbol(visitor);
            if (quotationLevel == 0) {
                if (signature) {
                    visitor.leaveSignature(quotationLevel, textRange());
                } else {
                    visitor.leaveText(quotationLevel, textRange());
                }
                textBegin = end;
            } else {
                visitor.leaveQuotation(quotationLevel, textRange());
                textBegin = end;
                quotationLevel = 0;
            }
            return State::NewLine;
        default:
            visitLeaveSymbol(visitor);
            return State::Text;
    }
}

void Parser::visitLeaveSymbol(AbstractVisitor& visitor) {
    visitor.visitSymbol(quotationLevel, current);
    visitor.leaveSymbol(quotationLevel, symbolRange());
}

} // namespace mail_quote
} // namespace msg_body
