#include <algorithm>
#include <iostream>
#include <unordered_map>
#include <internal/mail_quote/token.h>

namespace msg_body {
namespace mail_quote {

const Token Token::ampersand(Type::Ampersand, "&");
const Token Token::carriageReturn(Type::CarriageReturn, "\r");
const Token Token::greaterThan(Type::GreaterThan, ">");
const Token Token::lessThan(Type::LessThan, "<");
const Token Token::lineDelimiter(Type::LineDelimiter, "\r\n");
const Token Token::space(Type::Space, " ");
const Token Token::tab(Type::Tab, "\t");
const Token Token::signatureSeparator(Type::SignatureSeparator, "-- \r\n");

bool Token::isSpace() const {
    static const auto isSpace = [] (char value) {
        return std::isspace(value);
    };
    return !value_.empty() && std::all_of(value_.begin(), value_.end(), isSpace);
}

Token::Type Token::makeType(const std::string& value) {
    static const std::unordered_map<std::string, Type> tokens = {
        {ampersand.value(), ampersand.type()},
        {carriageReturn.value(), carriageReturn.type()},
        {greaterThan.value(), greaterThan.type()},
        {lessThan.value(), lessThan.type()},
        {lineDelimiter.value(), lineDelimiter.type()},
        {space.value(), space.type()},
        {tab.value(), tab.type()},
        {signatureSeparator.value(), signatureSeparator.type()},
        {"\n", Type::LineDelimiter},
        {"-- \n", signatureSeparator.type()},
        {"--\r\n", signatureSeparator.type()},
        {"--\n", signatureSeparator.type()},
    };
    const auto it = tokens.find(value);
    return it == tokens.end() ? Type::Symbols : it->second;
}

} // namespace mail_quote
} // namespace msg_body
