#include <internal/mail_quote/visitor.h>

namespace msg_body {
namespace mail_quote {

void Visitor::visit(int quotationLevel, Iterator /*current*/) {
    currentQuotationLevel = quotationLevel;
    visited = Visited::Nothing;
    paragraphVisitor.visit();
}

void Visitor::leave(int quotationLevel, const Range &/*range*/) {
    tryLeaveText();
    tryLeaveSignature();
    tryLeaveQuotation(quotationLevel);
    paragraphVisitor.leave();
}

void Visitor::visitQuotation(int quotationLevel, Iterator /*current*/) {
    tryLeaveText();
    tryLeaveSignature();
    tryLeaveQuotation(quotationLevel);
    tryVisitQuotation(quotationLevel);
    paragraphVisitor.visitLine();
}

void Visitor::leaveQuotation(int /*quotationLevel*/, const Range& /*range*/) {
    paragraphVisitor.leaveLine();
}

void Visitor::visitText(int quotationLevel, Iterator /*current*/) {
    tryLeaveSignature();
    tryLeaveQuotation(quotationLevel);
    tryVisitText();
    paragraphVisitor.visitLine();
}

void Visitor::leaveText(int /*quotationLevel*/, const Range& /*range*/) {
    paragraphVisitor.leaveLine();
}

void Visitor::visitSignature(int quotationLevel, Iterator /*current*/) {
    tryLeaveText();
    tryLeaveQuotation(quotationLevel);
    tryVisitSignature();
    paragraphVisitor.visitLine();
}

void Visitor::leaveSignature(int /*quotationLevel*/, const Range& /*range*/) {
    paragraphVisitor.leaveLine();
}

void Visitor::visitSymbol(int /*quotationLevel*/, Iterator current) {
    paragraphVisitor.visitSymbol(current);
}

void Visitor::leaveSymbol(int /*quotationLevel*/, const Range& range) {
    paragraphVisitor.leaveSymbol(range);
}

void Visitor::tryVisitText() {
    if (visited == Visited::Nothing) {
        paragraphVisitor.visitParagraph();
        paragraphVisitor.visitText();
        visited = Visited::Text;
    }
}

void Visitor::tryLeaveText() {
    if (visited == Visited::Text) {
        paragraphVisitor.leaveText();
        paragraphVisitor.leaveParagraph();
        visited = Visited::Nothing;
    }
}

void Visitor::tryVisitSignature() {
    if (visited == Visited::Nothing) {
        paragraphVisitor.visitParagraph();
        paragraphVisitor.visitSignature();
        visited = Visited::Signature;
    }
}

void Visitor::tryLeaveSignature() {
    if (visited == Visited::Signature) {
        paragraphVisitor.leaveSignature();
        paragraphVisitor.leaveParagraph();
        visited = Visited::Nothing;
    }
}

void Visitor::tryVisitQuotation(int quotationLevel) {
    for (; currentQuotationLevel < quotationLevel; ++currentQuotationLevel) {
        paragraphVisitor.visitParagraph();
        paragraphVisitor.visitQuotation();
    }
}

void Visitor::tryLeaveQuotation(int quotationLevel) {
    for (; currentQuotationLevel > quotationLevel; --currentQuotationLevel) {
        paragraphVisitor.leaveQuotation();
        paragraphVisitor.leaveParagraph();
    }
}

} // namespace mail_quote
} // namespace msg_body
