#include <internal/pa_log.h>
#include <internal/vdirect.h>
#include <internal/server/request_context.h>
#include <internal/config.h>
#include <internal/exceptions.h>
#include <internal/message_body.h>
#include <internal/transformer_attributes.h>
#include <internal/inline_spoofer.h>
#include <internal/mulcagate/logging.h>
#include <yamail/data/reflection/reflection.h>

namespace msg_body {

inline bool getLogicalParam(const RequestContext& req, const std::string& paramName) {
    return req.getArg(paramName) == "1";
}

inline std::set<int> getTypesParam(const RequestContext& req) {
    std::set<int> res;
    const auto typesStr = req.getArg("types");
    if (!typesStr.empty()) {
        std::set<std::string> types;
        boost::algorithm::split(types, typesStr, boost::algorithm::is_any_of(","),
                boost::algorithm::token_compress_on);
        try {
            auto strToInt = &boost::lexical_cast<int, std::string>;
            std::transform(types.begin(), types.end(), std::inserter(res, res.end()), strToInt);
        } catch (const boost::bad_lexical_cast&) {
            throw MessageBodyParamsException("cannot cast types in string: '" + typesStr + "'");
        }
    }
    return res;
}

MessageBodyParams parseMessageBodyParams(const RequestContext& req) {
    MessageBodyParams res;

    res.ticket = tvm::Ticket().setUid(req.getArg("uid")).setTicket(req.getHeader(tvm::headerName()));
    res.authDomain = req.getArg("auth_domain");
    res.userDomain = req.getArg("user_domain");
    if (res.ticket.uid().empty()) {
        throw MessageBodyParamsException("uid must not be empty");
    }

    res.mid = req.getArg("mid");
    if (res.mid.empty()) {
        throw MessageBodyParamsException("mid must not be empty");
    }

    res.hid = req.getArg("hid");
    if (res.hid.empty()) {
        res.hid = rootHid;
    }

    res.timeZone = req.getArg("tz");
    if (res.timeZone.empty()) {
        const std::string defaultTimeZone = "Europe/Moscow";
        res.timeZone = defaultTimeZone;
    }

    res.charset = req.getArg("charset");
    res.to = req.getArg("to");
    res.lang = req.getArg("lang");
    res.types = getTypesParam(req);
    res.flags = req.getArg("flags");

    res.isSecure = getLogicalParam(req, "secure");
    res.isSpam = getLogicalParam(req, "spam");

    res.remoteIp = req.remoteIp();
    res.requestId = req.requestId();
    res.connectionId = req.getArg("connection_id");
    res.yandexUidCookie = req.getArg("yandexuid");
    res.iCookie = req.getArg("icookie");
    res.clientType = req.getHeader("X-Yandex-ClientType");
    res.clientVersion = req.getHeader("X-Yandex-ClientVersion");
    res.userAgent = req.getArg("ua");
    res.testBuckets = req.getHeader("X-Yandex-ExpBoxes");
    res.enabledTestBuckets = req.getHeader("X-Yandex-EnabledExpBoxes");

    auto sanitizerStrategyParam = req.getOptionalArg("sanitizer_strategy");
    res.sanitizerStrategy = !sanitizerStrategyParam ? sanitizer::Strategy::real
        : yamail::data::reflection::from_string<sanitizer::Strategy>(*sanitizerStrategyParam);

    res.system = req.getOptionalArg("system");

    return res;
}

VdirectParams getVdirectParams(const MessageBodyParams& params) {
    VdirectParams res;
    res.uid = params.ticket.uid();
    res.authDomain = params.authDomain;
    res.userDomain  = params.userDomain;
    res.isSecure = params.isSecure;
    return res;
}

TransformerAttributes getTransformerAttributes(const MessageBodyParams& params) {
    TransformerAttributes res;

    res.uid = params.ticket.uid();
    res.useCharset = params.charset;
    res.mid = params.mid;
    res.to = params.to;
    res.lang = params.lang;
    res.types = params.types;
    res.requestId = params.requestId;
    res.sanitizerStrategy = params.sanitizerStrategy;
    res.system = params.system;

    res.setFlags(params.flags);

    return res;
}

JournalParams getJournalParams(const MessageBodyParams& params) {
    JournalParams res;   
    res.expBoxes = params.testBuckets;
    res.enabledExpBoxes = params.enabledTestBuckets;
    res.yandexUid = params.yandexUidCookie;
    res.iCookie = params.iCookie;
    res.clientType = params.clientType;
    res.clientVersion = params.clientVersion;
    res.userAgent = params.userAgent;
    res.connectionId = params.connectionId;
    return res;
}

MacsParams getMacsParams(const MessageBodyParams& params) {
    MacsParams res;
    res.uid = params.ticket.uid();
    res.realIp = params.remoteIp;
    res.requestId = params.requestId;
    return res;
}

DariaResult formMessageBody(const Configuration& config, const MessageBodyParams& params,
                            const http::headers& headersToPass, LogPtr logger, YieldCtx yc) {
    const PaLog paLog(__FUNCTION__, params.requestId);
    auto attrs = getTransformerAttributes(params);
    const auto mgLogger = MulcagateLog::create(params.requestId);

    try {
        const auto storageService = config.mailStorage->createService(mgLogger, params.ticket,
            params.requestId);

        const auto getClusterClient = makeGetClusterClient();
        auto getServiceTicket = makeGetServiceTicket(getTvmModule());

        Sanitizer sanitizer(config.sanitizer, params.isSecure, params.isSpam, params.requestId,
            headersToPass, std::move(getServiceTicket), yc, getClusterClient);
        VdirectPtr vdirect = getVdirect(config, getVdirectParams(params), !attrs.flags.noVdirectLinksWrap);

        const auto journal = getJournalParams(params);
        const auto asyncMacsService = getAsyncMacsService(*config.maildb, getMacsParams(params),
            journal, logger, yc);

        auto [messageAccessParams, attachesInfo] = mail_getter::getMessageAccessParamsWithAttaches(*asyncMacsService, params.mid);
        auto messageAccess = storageService->createMessageAccess(std::move(messageAccessParams), *config.recognizer, yc);
        const InlineSpooferPtr inlineSpoofer = getInlineSpoofer(config, logger, attrs,
            storageService, *messageAccess, yc);

        const auto asyncFactExtractor = TextAsyncFactExtractor(logger, config.factexParserConfig,
            config.factexConfig, params.requestId, yc, getClusterClient);

        DariaViewMaker dariaViewMaker(config,
            attrs,
            logger,
            *messageAccess,
            *config.contentTypeDetector,
            config.aliasClassList,
            config.bigLettersTrimThreshold,
            params.timeZone,
            vdirect,
            sanitizer,
            asyncMacsService,
            *inlineSpoofer,
            *config.recognizer,
            asyncFactExtractor,
            std::move(attachesInfo));
        const auto result = dariaViewMaker.createDariaMessage(params.hid);
        paLog.write();
        return result;
    } catch (const std::bad_alloc&) {
        throw;
    } catch (const mail_getter::UnknownMid&) {
        throw;
    } catch (const mail_getter::MessageNotFound&) {
        throw;
    } catch (const boost::system::system_error& e) {
        throw;
    } catch (const std::exception& e) {
        throw MessageFormingException(std::string("error in forming message: ") + e.what());
    }
}

}
