#include <internal/message_text.h>

#include <cstdint>
#include <boost/lexical_cast.hpp>
#include <boost/range/adaptor/filtered.hpp>

#include <butil/StrUtils/Encoding.h>
#include <butil/StrUtils/StrUtils.h>
#include <macs/envelope.h>
#include <mail_getter/recognizerWrapper.h>
#include <mail_getter/UTFizer.h>
#include <internal/config.h>
#include <internal/exceptions.h>
#include <internal/macs/factory.h>
#include <internal/mulcagate/logging.h>
#include <internal/server/request_context.h>
#include <internal/message_tree.h>
#include <internal/message_walker.h>
#include <internal/sanitizer_tykva.h>
#include <internal/logger.h>
#include <internal/trim.h>

namespace msg_body {

MessageTextParams parseMessageTextParams(const RequestContext& req) {
    MessageTextParams res;
    res.uid = req.getRequiredNonEmptyArg("uid");
    res.mid = req.getRequiredNonEmptyArg("mid");
    res.service = req.getOptionalArg("service");
    res.requestId = req.requestId();
    if (auto maxSizeArg = req.getOptionalArg("max_size")) {
        std::uint32_t maxSize = 0;
        if (!boost::conversion::try_lexical_convert(*maxSizeArg, maxSize)) {
            throw ParamsException("max_size");
        }

        res.maxSize = maxSize;
    }
    return res;
}

MacsParams getMacsParams(const MessageTextParams& params) {
    MacsParams res;
    res.uid = params.uid;
    res.requestId = params.requestId;
    return res;
}

MessageText getMessageText(const Configuration& config, const MessageTextParams& params, LogPtr logger, YieldCtx yield) {
    const auto asyncMacsService = getAsyncMacsService(*config.maildb, getMacsParams(params), JournalParams{}, logger, yield);
    const auto messageAccessParams = mail_getter::getMessageAccessParams(*asyncMacsService, params.mid);
    const auto mgLogger = MulcagateLog::create(params.requestId);
    const auto storageService = config.mailStorage->createService(mgLogger, tvm::Ticket(), params.requestId);
    auto messageAccess = storageService->createMessageAccess(std::move(messageAccessParams), *config.recognizer, yield);

    MessageTreeCreator messageTreeCreator(*messageAccess, *config.contentTypeDetector, macs::Attachments{});

    std::string hid = rootHid;
    MessageTree messageTree = messageTreeCreator.create(hid);
    MessageWalker messageWalker;
    messageWalker.make(messageTree);

    auto textParts = messageWalker.parts() | boost::adaptors::filtered([](const auto &part) {
        return ((part.metaType == "text") && !(part.headerStruct.contentDisposition() == "attachment"));
    });

    int currentPart = 0;
    const int maxParts = 1;

    MessageText messageText;
    for (const auto &part : textParts) {
        const auto partLength = (part.headerStruct.offsetEnd() - part.headerStruct.offsetBegin());

        MessageTextPartMode mode = MessageTextPartMode::full;
        std::string content;
        std::optional<std::string> lang;
        if ((0 != config.messageTextConfig.maxPartLength) && (partLength <= config.messageTextConfig.maxPartLength)) {
            ++currentPart;
            content = messageAccess->getBody(part.hid);

            Encoding::decode(part.headerStruct.encoding(), content);

            const auto& charset = part.headerStruct.charset();
            const auto langId = utfizeStringDetectLang(*config.recognizer, content, charset);
            if (Recognizer::LANG_UNK != langId) {
                lang = config.recognizer->isoNameByLanguageCode(langId);
            }

            TStrUtils::replaceBadSymbols(content);

            const bool needTransform = (!content.empty() && (part.contentType.subtype() == "html"));
            if (needTransform) {
                sanitizer::sanitizeByTykva(content);
            }

            if (params.maxSize && trimBigLetter(&content, *params.maxSize)) {
                mode = MessageTextPartMode::trimmed;
            }
        } else {
            auto envelope = asyncMacsService->getById(params.mid);

            content = envelope.firstline();
            mode = MessageTextPartMode::firstline;
            MBODY_LOG_INFO(logger, log::where_name="message/text", log::message="text part too long, return firstline");
        }

        messageText.textParts.push_back(MessageText::Part{mode, std::move(content), std::move(lang)});

        if ((0 != maxParts) && (currentPart <= maxParts)) {
            break;
        }
    }

    return messageText;
}

} // namespace msg_body
