#include <boost/filesystem.hpp>

#include <butil/butil.h>
#include <macs/mime_part_factory.h>
#include <mail_getter/MessageAccess.h>
#include <mail_getter/alias_class_list.h>

#include <internal/transformers/routines.h>
#include <internal/message_tree.h>

namespace msg_body {

MessageTreeCreator::MessageTreeCreator(MessageAccess& messageAccess,
        IContentTypeDetector& contentTypeDetector, macs::Attachments attachments)
        : messageAccess_(messageAccess)
        , contentTypeDetector_(contentTypeDetector)
        , attachments_(std::move(attachments)) {
}

size_t localHidPart(const std::string& hid) {
    const size_t lastDot = hid.find_last_of('.');
    try {
        return lastDot == std::string::npos
            ? boost::lexical_cast<size_t>(hid)
            : boost::lexical_cast<size_t>(hid.substr(lastDot+1));
    } catch (boost::bad_lexical_cast&) {
        throw MessageTreeException("Invalid hid: " + hid);
    }
}

MetaAttributes getMessageHeaderParsedFromBody(MessageAccess& messageAccess,
        const std::string& hid) {
    const std::string body = messageAccess.getBody(hid);
    return mail_getter::getMessageHeaderParsed(body);
}

MimeType getMimeType(MetaPart& hs) {
    try {
        if (hs.contentType().empty() || hs.contentSubtype().empty()) {
            return MimeType();
        }
        return MimeType(hs.contentType(), hs.contentSubtype());
    } catch (const std::exception&) {
        return MimeType();
    }
}

MimeType MessageTreeCreator::detectContentType(const std::string& fileName) const {
    MimeType res = contentTypeDetector_.detect(fileName);
    ContentTypeDetector::adjustType(res);
    return res;
}

void MessageTreeCreator::detectAttrsContentType(MessagePart& messagePart) const {
    auto& hs = messagePart.headerStruct;
    const std::string& fileName = hs.name();
    if (!fileName.empty()) {
        const MimeType contentType = detectContentType(fileName);
        if (!contentType.isDefaultMimeType()) {
            hs = macs::MimePartFactory(hs).contentType(contentType.type())
                    .contentSubtype(contentType.subtype()).release();
            messagePart.contentType = contentType;
        }
    }
}

void MessageTreeCreator::fillAttachmentSize(const std::string& hid, MessagePart& messagePart) const {
    const auto attachIt = std::find_if(attachments_.begin(), attachments_.end(), [&hid] (const macs::AttachmentDescriptor& attach) {
        return attach.m_hid == hid;
    });
    if (attachments_.end() != attachIt) {
        messagePart.length = attachIt->m_size;
    }
}

void MessageTreeCreator::fillPart(const std::string& hid, MessagePart& messagePart) const {
    messagePart.hid = hid;
    messagePart.headerStruct = messageAccess_.getHeaderStruct(hid);
    messagePart.bodyStruct = messageAccess_.getBodyStruct(hid);
    messagePart.contentType = getMimeType(messagePart.headerStruct);
    fillAttachmentSize(hid, messagePart);

    messagePart.stid = messageAccess_.getStId();
    detectAttrsContentType(messagePart);

    if (messagePart.contentType.type() == "message" && messagePart.contentType.subtype() == "rfc822") {
        messagePart.messageHeader =  messageAccess_.getMessageHeaderParsed(hid + ".1");
    }

    if (messagePart.contentType.type() == "text" && messagePart.contentType.subtype() == "rfc822-headers") {
        messagePart.messageHeader = getMessageHeaderParsedFromBody(messageAccess_, hid);
    }
}

void MessageTreeCreator::processTree(const std::string& hid, MessageTree& messagePart, size_t level) const {
    static const size_t maxPartDepth = 100;
    if (level >= maxPartDepth) {
        throw MessageTreeException("reached max depth of message tree part processing: " +
                                   boost::lexical_cast<std::string>(level));
    }
    fillPart(hid, messagePart.part);
    const MetaLevel& metaLevel = messagePart.part.bodyStruct;
    for (MetaLevel::const_iterator childHid = metaLevel.begin(); childHid != metaLevel.end(); ++childHid) {
        boost::shared_ptr<MessageTree> childPart(new MessageTree);
        processTree(*childHid, *childPart, level+1);
        messagePart.children[localHidPart(*childHid)] = childPart;
    }
}

MessageTree MessageTreeCreator::create(const std::string& hid) const {
    const std::string startHid = hid.empty() ? rootHid : hid;
    MessageTree messageTree;
    processTree(startHid, messageTree, 0);
    return messageTree;
}

MessagePartTypeInfo getTypeInfo(MessagePart& part, const IContentTypeDetector& contentTypeDetector,
        const AliasClassList& aliasClassList) {
    MessagePartTypeInfo res;
    res.contentType = part.contentType;

    const auto& hs = part.headerStruct;

    res.dispositionValue = hs.contentDisposition();
    if (res.dispositionValue.empty()) {
        res.dispositionValue = "attachment";
    }
    res.contentId = hs.cid();

    res.dispositionFilename = hs.fileName();

    const std::string& name = (!res.dispositionFilename.empty()) ? res.dispositionFilename : hs.name();
    part.realName = name;
    res.name = name.empty() ? "no_name" : name;

    res.nameUriEncoded = encode_url(name);
    boost::filesystem::path path(name);
    res.fileName = path.stem().string();
    res.fileExtension = path.extension().string();

    const std::string& type = part.contentType.type();
    const std::string& subtype = part.contentType.subtype();
    res.partClassInfo = determinePartClass(contentTypeDetector, aliasClassList, name, type, subtype);
    if (part.length.has_value()) {
        res.length = *part.length;
    } else {
        res.length = estimateSize(hs.length(), hs.encoding());
    }
    return res;
}

}
