#include <boost/bind.hpp>
#include <boost/range/algorithm/for_each.hpp>
#include <boost/range/adaptors.hpp>

#include <internal/message_walker.h>

namespace msg_body {

void MessageWalker::makeMessage(MessageTree& tree) {
    tree.part.metaType = "message";
    parts_.push_back(tree.part);
}

void MessageWalker::makeSimplePart(MessageTree& tree) {
    tree.part.metaType = classifyMetatype(tree.part);
    parts_.push_back(tree.part);
}

void MessageWalker::makeAltCalendarChildPart(const MessageTree::Children::value_type& node) {
    const MessageTree& tree = *node.second;
    MessagePart part = tree.part;
    part.isMixedAttach = false;
    part.metaType = classifyMetatype(tree.part);
    parts_.emplace_back(std::move(part));
}

bool isValidCalendar(const MessageTree::Children::value_type node) {
    return node.second->isValid &&
            node.second->part.contentType == MimeType("text", "calendar");
}

void MessageWalker::makeChildrenCalendars(MessageTree& tree) {
    MessageTree::Children& children = tree.children;
    boost::for_each(children | boost::adaptors::filtered(isValidCalendar),
        boost::bind(&MessageWalker::makeAltCalendarChildPart, this, _1));
}

bool isAlternativeSupported(const MimeType& mimeType) {
    return mimeType.type() == "multipart"
        || mimeType.type() == "image"
        || mimeType == MimeType("text", "plain")
        || mimeType == MimeType("text", "html")
        || mimeType == MimeType("message", "delivery-status");
}

bool isEmptyChild(const MessagePart& part) {
    return part.headerStruct.length() == 0;
}

bool isValidAlternativeChild(const MessageTree::Children::value_type& v) {
    const MessagePart& part = v.second->part;
    return v.second->isValid &&
            isAlternativeSupported(part.contentType) && !isEmptyChild(part)
        && part.contentType != MimeType("text", "calendar");
}

void MessageWalker::makeAlternativeChild(MessageTree& tree) {
    const MessageTree::Children& children = tree.children;
    MessageTree::Children::const_reverse_iterator altChild =
        std::find_if(children.rbegin(), children.rend(), &isValidAlternativeChild);
    return altChild == children.rend()
        ? makeMixed(tree)
        : makeChildPart(*altChild);
}

void MessageWalker::makeAlternative(MessageTree& tree) {
    makeChildrenCalendars(tree);
    makeAlternativeChild(tree);
}

void MessageWalker::makeSigned(MessageTree& tree) {
    if (tree.part.bodyStruct.size() == 2) {
        tree.part.metaType = "signed";
        parts_.push_back(tree.part);
        makePart(*tree.children[1]);
    } else {
        makeMixed(tree);
    }
}

void MessageWalker::makeMixed(MessageTree& tree) {
    boost::for_each(tree.children, boost::bind(&MessageWalker::makeChildPart, this, _1));
}

void MessageWalker::makeMultipart(MessageTree& tree) {
    const MessagePart& part = tree.part;
    if (part.contentType.subtype() == "alternative") {
        makeAlternative(tree);
    } else if (part.contentType.subtype() == "signed") {
        makeSigned(tree);
    } else {
        makeMixed(tree);
    }
}

void MessageWalker::makePart(MessageTree& tree) {
    const MessagePart& part = tree.part;
    if (part.contentType.type() == "multipart") {
        makeMultipart(tree);
    } else if (part.contentType == MimeType("message", "rfc822")) {
        makeMessage(tree);
    } else {
        makeSimplePart(tree);
    }
}

void MessageWalker::makeChildPart(const MessageTree::Children::value_type& node) {
    makePart(*node.second);
}

bool MessageWalker::markInvalidChildNodes(MessageTree::Children::value_type& node,
        const std::string& invalidPartHid) {
    return markInvalidNodes(*node.second, invalidPartHid);
}

bool MessageWalker::hasInvalidChildren(MessageTree& tree, const std::string& invalidPartHid) {
    MessageTree::Children& children = tree.children;
    return std::find_if(children.begin(), children.end(),
            boost::bind(&MessageWalker::markInvalidChildNodes,
                    this, _1, invalidPartHid)) != children.end();
}

bool MessageWalker::hasValidAlternativeChild(MessageTree& tree) {
    MessageTree::Children& children = tree.children;
    return tree.part.contentType == MimeType("multipart", "alternative") &&
            std::find_if(children.rbegin(), children.rend(),
                    &isValidAlternativeChild) != children.rend();
}

bool MessageWalker::markInvalidNodes(MessageTree& tree, const std::string& invalidPartHid) {
    if (tree.part.hid == invalidPartHid ||
            (hasInvalidChildren(tree, invalidPartHid) && !hasValidAlternativeChild(tree))) {
        tree.isValid = false;
        return true;
    }
    return false;
}

bool MessageWalker::make(MessageTree& tree, const std::string& invalidPartHid) {
    markInvalidNodes(tree, invalidPartHid);
    if (!tree.isValid) {
        return false;
    }
    makePart(tree);
    return true;
}

void MessageWalker::make(MessageTree& tree) {
    makePart(tree);
}

const MessageParts& MessageWalker::parts() const {
    return parts_;
}

}
